-- tracker_guild.lua
-- WhoDAT - Guild Bank & Roster Tracking
-- Tracks guild bank contents, money, roster members, and changes over time
-- Wrath 3.3.5a compatible
-- 
-- VERSION: 3.0.1 - FIXED DEDUPLICATION
-- Changes:
-- - Improved hash generation using full timestamps
-- - Better handling of incomplete date data from WoW API
-- - Enhanced deduplication logic
-- - More detailed logging

local ADDON_NAME = "WhoDAT"
local NS = _G[ADDON_NAME] or {}
_G[ADDON_NAME] = NS

-- ============================================================================
-- Module State & Configuration
-- ============================================================================

NS.Guild = NS.Guild or {}
local Guild = NS.Guild

local guildState = {
  lastBankScan = 0,
  lastRosterScan = 0,
  lastMoneyUpdate = 0,
  lastTransactionLogQuery = 0,
  lastMoneyLogQuery = 0,
  bankOpen = false,
  previousRoster = {},
  previousBankContents = {},
  previousMoney = 0,
}

-- Throttle settings (seconds)
local THROTTLE = {
  bank_scan = 2,              -- Don't scan bank more than once per 2 seconds
  roster_scan = 30,           -- Roster updates every 30 seconds max
  money_update = 1,           -- Money updates every second
  transaction_log_query = 5,  -- Transaction log queries every 5 seconds
  money_log_query = 5,        -- Money log queries every 5 seconds
}

-- ============================================================================
-- Utility Functions
-- ============================================================================

local function now() return time() end

local function shouldThrottle(key, seconds)
  local stateKey = "last" .. key:gsub("^%l", string.upper):gsub("_(%l)", string.upper)
  local last = guildState[stateKey] or 0
  if (now() - last) < seconds then return true end
  guildState[stateKey] = now()
  return false
end

-- ============================================================================
-- Wrath-Compatible Timer (C_Timer doesn't exist in 3.3.5a)
-- ============================================================================
local function DelayedCall(seconds, callback)
  local frame = CreateFrame("Frame")
  local elapsed = 0
  
  frame:SetScript("OnUpdate", function(self, delta)
    elapsed = elapsed + delta
    if elapsed >= seconds then
      frame:SetScript("OnUpdate", nil)
      callback()
    end
  end)
end

local function getCharacterKey()
  if NS.Utils and NS.Utils.GetPlayerKey then
    return NS.Utils.GetPlayerKey()
  end
  return (GetRealmName() or "Unknown") .. ":" .. (UnitName("player") or "Player")
end

local function ensureGuildData()
  local key = getCharacterKey()
  WhoDatDB = WhoDatDB or {}
  WhoDatDB.characters = WhoDatDB.characters or {}
  WhoDatDB.characters[key] = WhoDatDB.characters[key] or {}
  
  local C = WhoDatDB.characters[key]
  C.guild = C.guild or {}
  C.guild.bank = C.guild.bank or {}
  C.guild.roster = C.guild.roster or {}
  C.guild.events = C.guild.events or {}
  C.guild.snapshots = C.guild.snapshots or {}
  C.guild.logs = C.guild.logs or {}
  C.guild.logs.transactions = C.guild.logs.transactions or {}
  C.guild.logs.money = C.guild.logs.money or {}
  
  return key, C
end

local function safeGetItemInfo(link)
  if not link then return nil end
  
  -- Try to use Utils if available
  if NS.Utils and NS.Utils.GetItemMeta then
    local name, ilvl, quality, _, _, _, _, stack, equipLoc, icon, itemID = NS.Utils.GetItemMeta(link)
    return {
      name = name,
      id = itemID,
      link = link,
      quality = quality,
      ilvl = ilvl,
      stack_size = stack,
      icon = icon,
      equip_loc = equipLoc,
    }
  end
  
  -- Fallback to basic GetItemInfo
  local name, _, quality, ilvl, reqLevel, class, subClass, stack, equipLoc, icon, sellPrice, itemID = GetItemInfo(link)
  if not name then return nil end
  
  return {
    name = name,
    id = itemID,
    link = link,
    quality = quality,
    ilvl = ilvl,
    stack_size = stack,
    icon = icon,
    equip_loc = equipLoc,
  }
end

-- ============================================================================
-- IMPROVED: Hash Generation for Deduplication
-- ============================================================================
-- This function generates a unique hash for each transaction that will be
-- used to prevent duplicates when uploading data multiple times.
-- 
-- KEY IMPROVEMENT: Uses full timestamp instead of year/month/day/hour
-- because the WoW API often returns year=0 and month=0, making hashes less unique.
-- ============================================================================

local function generateTransactionHash(trans)
  -- Create a stable hash from transaction details that persists across uploads
  -- Using timestamp provides better uniqueness than year/month/day/hour separately
  local parts = {
    tostring(trans.player or "Unknown"),
    tostring(trans.type or "deposit"),
    tostring(trans.amount or trans.item_id or "0"),
    tostring(trans.count or 1),
    tostring(trans.tab or 0),
    tostring(trans.ts or 0),  -- Using full timestamp for better uniqueness
  }
  return table.concat(parts, "|")
end

-- ============================================================================
-- Guild Module Initialization
-- ============================================================================

function Guild:IsEnabled()
  return true
end

function Guild:HasBankLogAccess()
  if not IsInGuild() then return false end
  
  local guildName, _, rankIndex = GetGuildInfo("player")
  if not guildName then return false end
  
  -- Guild Master (rank 0) always has access
  if rankIndex == 0 then return true end
  
  -- Check if player has VIEW_TRANSACTION permission
  local canView = false
  for i = 1, GetNumGuildBankTabs() do
    local name, icon, isViewable, canDeposit, numWithdrawals, remainingWithdrawals = GetGuildBankTabInfo(i)
    if isViewable then
      canView = true
      break
    end
  end
  
  return canView
end

-- ============================================================================
-- Guild Bank Scanning
-- ============================================================================

function Guild:ScanGuildBank()
  if not self:IsEnabled() then return end
  if not IsInGuild() then return end
  if shouldThrottle("bank_scan", THROTTLE.bank_scan) then return end
  
  local key, C = ensureGuildData()
  local ts = now()
  
  -- Get guild bank money (in copper)
  local money = GetGuildBankMoney and GetGuildBankMoney() or 0
  
  -- Scan all accessible tabs
  local tabs = {}
  local numTabs = GetNumGuildBankTabs and GetNumGuildBankTabs() or 0
  
  for tabIndex = 1, numTabs do
    local tabName, tabIcon, isViewable, canDeposit, numWithdrawals, remainingWithdrawals = GetGuildBankTabInfo(tabIndex)
    
    -- Only process tabs we can view
    if isViewable then
      local tab = {
        index = tabIndex,
        name = tabName or ("Tab " .. tabIndex),
        icon = tabIcon,
        can_deposit = canDeposit,
        withdrawals_remaining = remainingWithdrawals or 0,
        slots = {},
      }
      
      -- Scan all slots in this tab (98 slots per tab in Wrath)
      for slotIndex = 1, 98 do
        local link = GetGuildBankItemLink(tabIndex, slotIndex)
        
        if link then
          local texture, count, locked = GetGuildBankItemInfo(tabIndex, slotIndex)
          local itemInfo = safeGetItemInfo(link)
          
          if itemInfo then
            table.insert(tab.slots, {
              slot = slotIndex,
              link = link,
              id = itemInfo.id,
              name = itemInfo.name,
              icon = texture or itemInfo.icon,
              count = count or 1,
              quality = itemInfo.quality,
              ilvl = itemInfo.ilvl,
              locked = locked,
            })
          end
        end
      end
      
      table.insert(tabs, tab)
    end
  end
  
  -- Store current snapshot
  C.guild.snapshots.bank = {
    ts = ts,
    money = money,
    tabs = tabs,
    num_tabs = numTabs,
  }
  
  -- Detect money changes
  if money ~= guildState.previousMoney and guildState.previousMoney > 0 then
    local moneyDelta = money - guildState.previousMoney
    
    table.insert(C.guild.events, {
      ts = ts,
      type = "money_change",
      old_amount = guildState.previousMoney,
      new_amount = money,
      delta = moneyDelta,
    })
    
    if NS.Log then
      NS.Log("INFO", "Guild bank money changed: %s copper (%+d)", money, moneyDelta)
    end
  end
  
  guildState.previousMoney = money
end

-- ============================================================================
-- Guild Roster Scanning
-- ============================================================================

function Guild:ScanRoster()
  if not self:IsEnabled() then return end
  if not IsInGuild() then return end
  if shouldThrottle("roster_scan", THROTTLE.roster_scan) then return end
  
  local key, C = ensureGuildData()
  local ts = now()
  
  GuildRoster()
  
  local numTotal, numOnline, numMobileOnline = GetNumGuildMembers()
  local members = {}
  
  for i = 1, numTotal do
    local name, rank, rankIndex, level, class, zone, note, officernote, online, status, classFileName, achievementPoints, achievementRank, isMobile, canSoR, repStanding = GetGuildRosterInfo(i)
    
    if name then
      -- Remove realm from name if present
      local playerName = strsplit("-", name)
      
      table.insert(members, {
        name = playerName,
        full_name = name,
        rank = rank,
        rank_index = rankIndex,
        level = level,
        class = class,
        class_file = classFileName,
        zone = zone,
        note = note,
        officer_note = officernote,
        online = online,
        status = status,
        achievement_points = achievementPoints,
        is_mobile = isMobile,
      })
    end
  end
  
  -- Store snapshot
  C.guild.snapshots.roster = {
    ts = ts,
    num_members = numTotal,
    num_online = numOnline,
    members = members,
  }
  
  -- Detect roster changes
  if #guildState.previousRoster > 0 then
    local added, removed = {}, {}
    local prevByName = {}
    
    for _, m in ipairs(guildState.previousRoster) do
      prevByName[m.name] = true
    end
    
    local currentByName = {}
    for _, m in ipairs(members) do
      currentByName[m.name] = true
      if not prevByName[m.name] then
        table.insert(added, m.name)
      end
    end
    
    for _, m in ipairs(guildState.previousRoster) do
      if not currentByName[m.name] then
        table.insert(removed, m.name)
      end
    end
    
    if #added > 0 or #removed > 0 then
      table.insert(C.guild.events, {
        ts = ts,
        type = "roster_change",
        num_added = #added,
        num_removed = #removed,
        added_members = added,
        removed_members = removed,
      })
      
      if NS.Log then
        NS.Log("INFO", "Roster changed: +%d -%d", #added, #removed)
      end
    end
  end
  
  guildState.previousRoster = members
end

-- ============================================================================
-- IMPROVED: Transaction Log Parsing
-- ============================================================================

local function parseItemTransaction(index, tabIndex)
  local transType, playerName, itemLink, count, year, month, day, hour = 
    GetGuildBankTransaction(tabIndex, index)
  
  if not transType then return nil end
  
  -- Extract item_id from link
  local itemId = nil
  local itemName = nil
  if itemLink then
    if string.match(itemLink, "item:(%d+)") then
      itemId = tonumber(string.match(itemLink, "item:(%d+)"))
    end
    
    -- Try to get item name
    if NS.Utils and NS.Utils.GetItemMeta then
      itemName = NS.Utils.GetItemMeta(itemLink)
    else
      itemName = GetItemInfo(itemLink)
    end
  end
  
  -- SIMPLIFIED TIMESTAMP LOGIC (same as money transactions)
  local timestamp
  if year and year > 0 and month and month > 0 and day and hour then
    timestamp = time({
      year = year,
      month = month,
      day = day,
      hour = hour,
      min = 0,
      sec = 0
    })
  else
    local now = time()
    timestamp = now - ((index - 1) * 60)
  end
  
  local trans = {
    ts = timestamp,
    type = transType,
    player = playerName or "Unknown",
    item_id = itemId,
    item_name = itemName or "Unknown Item",
    item_link = itemLink,
    count = count or 1,
    tab = tabIndex,
    year = year or 0,
    month = month or 0,
    day = day or 0,
    hour = hour or 0,
  }
  
  -- Generate hash using improved method
  trans.hash = generateTransactionHash(trans)
  
  return trans
end

local function parseMoneyTransaction(index)
  local transType, playerName, amount, year, month, day, hour = 
    GetGuildBankMoneyTransaction(index)
  
  if not transType then return nil end
  
  -- SIMPLIFIED TIMESTAMP LOGIC:
  -- The WoW API often returns year=0 and month=0 for guild bank logs,
  -- making date reconstruction unreliable. Instead, we use a simpler approach:
  --
  -- 1. If we have complete valid date data (year>0, month>0), use it
  -- 2. Otherwise, use current time minus position offset
  --    (This preserves ordering: transaction 1 is newer than transaction 2)
  
  local timestamp
  if year and year > 0 and month and month > 0 and day and hour then
    -- We have complete date data - use it
    timestamp = time({
      year = year,
      month = month,
      day = day,
      hour = hour,
      min = 0,
      sec = 0
    })
  else
    -- Incomplete or missing date data
    -- Use current time minus position offset to preserve ordering
    -- Transaction at index 1 (most recent) gets current time
    -- Transaction at index 2 gets current time - 60 seconds, etc.
    local now = time()
    timestamp = now - ((index - 1) * 60)  -- 1 minute apart
  end
  
  local trans = {
    ts = timestamp,
    type = transType,
    player = playerName or "Unknown",
    amount = amount or 0,
    year = year or 0,
    month = month or 0,
    day = day or 0,
    hour = hour or 0,
  }
  
  -- Generate hash using improved method
  trans.hash = generateTransactionHash(trans)
  
  return trans
end

-- ============================================================================
-- Guild Bank Transaction Logs Scanning
-- ============================================================================

function Guild:ScanBankTransactionLogs()
  if not self:IsEnabled() then return end
  if not IsInGuild() then return end
  if not self:HasBankLogAccess() then return end
  if shouldThrottle("transaction_log_query", THROTTLE.transaction_log_query) then return end
  
  local key, C = ensureGuildData()
  
  -- Create deduplication set
  local existing = {}
  for _, trans in ipairs(C.guild.logs.transactions) do
    local transKey = trans.hash or generateTransactionHash(trans)
    existing[transKey] = true
  end
  
  local numTabs = GetNumGuildBankTabs() or 0
  local totalNewLogs = 0
  
  for tabIndex = 1, numTabs do
    QueryGuildBankLog(tabIndex)
    
    local numTransactions = GetNumGuildBankTransactions(tabIndex) or 0
    
    if numTransactions > 0 then
      for i = 1, math.min(numTransactions, 25) do
        local trans = parseItemTransaction(i, tabIndex)
        
        if trans then
          local transKey = trans.hash
          
          if not existing[transKey] then
            table.insert(C.guild.logs.transactions, trans)
            existing[transKey] = true
            totalNewLogs = totalNewLogs + 1
          end
        end
      end
    end
  end
  
  -- Limit total log size
  while #C.guild.logs.transactions > 500 do
    table.remove(C.guild.logs.transactions, 1)
  end
  
  if NS.Log and totalNewLogs > 0 then
    NS.Log("INFO", "Transaction logs: %d new entries added", totalNewLogs)
  end
end

function Guild:ScanBankMoneyLog()
  if not self:IsEnabled() then return end
  if not IsInGuild() then return end
  if not self:HasBankLogAccess() then return end
  if shouldThrottle("money_log_query", THROTTLE.money_log_query) then return end
  
  local key, C = ensureGuildData()
  
  -- Create deduplication set
  local existing = {}
  for _, trans in ipairs(C.guild.logs.money) do
    local transKey = trans.hash or generateTransactionHash(trans)
    existing[transKey] = true
  end
  
  -- Query money log (use MAX_GUILDBANK_TABS + 1 for money log)
  QueryGuildBankLog(MAX_GUILDBANK_TABS + 1)
  
  local numTransactions = GetNumGuildBankMoneyTransactions() or 0
  local newLogsFound = 0
  
  if numTransactions > 0 then
    for i = 1, math.min(numTransactions, 25) do
      local trans = parseMoneyTransaction(i)
      
      if trans then
        local transKey = trans.hash
        
        if not existing[transKey] then
          table.insert(C.guild.logs.money, trans)
          newLogsFound = newLogsFound + 1
        end
      end
    end
  end
  
  -- Limit total log size
  while #C.guild.logs.money > 500 do
    table.remove(C.guild.logs.money, 1)
  end
  
  if NS.Log and newLogsFound > 0 then
    NS.Log("INFO", "Money logs: %d new entries added", newLogsFound)
  end
end

-- ============================================================================
-- Event Handlers
-- ============================================================================

function Guild:OnGuildBankOpened()
  guildState.bankOpen = true
  
  -- Query all tabs to request data from server
  local numTabs = GetNumGuildBankTabs() or 0
  
  -- Query each tab to load its data
  for tabIndex = 1, numTabs do
    QueryGuildBankTab(tabIndex)
  end
  
  -- Wait 5 seconds then scan
  DelayedCall(5, function()
    if guildState.bankOpen then
      self:ScanGuildBank()
      print("|cff00ff00[WhoDAT]|r Guild bank tabs scanned!")
    end
  end)
  
  -- Check log access and scan if available
  if self:HasBankLogAccess() then
    DelayedCall(2, function()
      self:ScanBankTransactionLogs()
      self:ScanBankMoneyLog()
    end)
  end
end

function Guild:OnGuildBankClosed()
  guildState.bankOpen = false
end

function Guild:OnGuildRosterUpdate()
  self:ScanRoster()
end

-- ============================================================================
-- Guild Bank Scan Button
-- ============================================================================

local guildBankScanButton = nil

local function CreateGuildBankScanButton()
  if guildBankScanButton then return end
  if not GuildBankFrame then return end
  
  -- Create button
  guildBankScanButton = CreateFrame("Button", "WhoDAT_GuildBankScanButton", GuildBankFrame, "UIPanelButtonTemplate")
  guildBankScanButton:SetWidth(120)
  guildBankScanButton:SetHeight(22)
  
  -- Position it
  guildBankScanButton:SetPoint("TOPRIGHT", GuildBankFrame, "BOTTOMRIGHT", -10, -5)
  guildBankScanButton:SetText("Capture Logs")
  
  -- On click: Capture a fresh snapshot of logs
  guildBankScanButton:SetScript("OnClick", function()
    local hasAccess = Guild:HasBankLogAccess()
    
    if not hasAccess then
      print("[WhoDAT] You need guild master or officer permissions to view logs")
      return
    end
    
    print("[WhoDAT] Capturing guild bank logs...")
    
    -- Reset throttles to allow immediate capture
    guildState.lastTransactionLogQuery = 0
    guildState.lastMoneyLogQuery = 0
    
    -- Clear previous logs to get a fresh snapshot
    -- This is INTENTIONAL - each capture should be a clean snapshot of the current 25 visible transactions
    local key = getCharacterKey()
    if WhoDatDB and WhoDatDB.characters and WhoDatDB.characters[key] then
      local C = WhoDatDB.characters[key]
      if C.guild and C.guild.logs then
        C.guild.logs.transactions = {}
        C.guild.logs.money = {}
      end
    end
    
    -- Scan logs
    Guild:ScanBankTransactionLogs()
    Guild:ScanBankMoneyLog()
    
    -- Show results
    local key = getCharacterKey()
    local C = WhoDatDB.characters[key]
    
    if C and C.guild and C.guild.logs then
      local transCount = #(C.guild.logs.transactions or {})
      local moneyCount = #(C.guild.logs.money or {})
      print(string.format("[WhoDAT] ✓ Captured %d item transactions, %d money transactions", transCount, moneyCount))
      
      if transCount > 0 or moneyCount > 0 then
        print("[WhoDAT] Logs saved! Export your data to upload them.")
        print("|cffff8800[TIP]|r Only click 'Capture Logs' once per upload to avoid confusion!")
      end
    end
  end)
  
  -- Tooltip
  guildBankScanButton:SetScript("OnEnter", function(self)
    GameTooltip:SetOwner(self, "ANCHOR_TOP")
    GameTooltip:SetText("WhoDAT Guild Bank Log Capture", 1, 1, 1, 1, true)
    GameTooltip:AddLine(" ", 1, 1, 1, true)
    
    local hasAccess = Guild:HasBankLogAccess()
    
    if hasAccess then
      GameTooltip:AddLine("Click to capture guild bank transaction logs", 1, 1, 1, true)
      GameTooltip:AddLine(" ", 1, 1, 1, true)
      GameTooltip:AddLine("Captures:", 0.7, 0.7, 0.7, true)
      GameTooltip:AddLine("• Item deposits and withdrawals", 1, 1, 1, true)
      GameTooltip:AddLine("• Money deposits and withdrawals", 1, 1, 1, true)
      GameTooltip:AddLine("• Repair costs", 1, 1, 1, true)
      GameTooltip:AddLine(" ", 1, 1, 1, true)
      GameTooltip:AddLine("TIP: Click 'Log' tab first to load data", 0.5, 1, 0.5, true)
      GameTooltip:AddLine("Then click this button ONCE per upload", 0.5, 1, 0.5, true)
    else
      GameTooltip:AddLine("You need guild master or officer", 0.8, 0.4, 0.4, true)
      GameTooltip:AddLine("permissions to capture logs", 0.8, 0.4, 0.4, true)
    end
    
    GameTooltip:Show()
  end)
  
  guildBankScanButton:SetScript("OnLeave", function(self)
    GameTooltip:Hide()
  end)
  
  -- Update button state based on permissions
  local hasAccess = Guild:HasBankLogAccess()
  if not hasAccess then
    guildBankScanButton:Disable()
    guildBankScanButton:SetAlpha(0.5)
  end
end

-- Hook to create button when guild bank opens
local function HookGuildBank()
  if not GuildBankFrame then return end
  
  if GuildBankFrame and not guildBankScanButton then
    DelayedCall(0.1, CreateGuildBankScanButton)
  end
end

-- Initialize button when guild bank is opened
local buttonInitFrame = CreateFrame("Frame")
buttonInitFrame:RegisterEvent("GUILDBANKFRAME_OPENED")
buttonInitFrame:SetScript("OnEvent", function(self, event)
  if event == "GUILDBANKFRAME_OPENED" then
    HookGuildBank()
  end
end)

-- ============================================================================
-- Event Registration
-- ============================================================================

local eventFrame = CreateFrame("Frame")
eventFrame:RegisterEvent("GUILDBANKFRAME_OPENED")
eventFrame:RegisterEvent("GUILDBANKFRAME_CLOSED")
eventFrame:RegisterEvent("GUILD_ROSTER_UPDATE")

eventFrame:SetScript("OnEvent", function(self, event, ...)
  if event == "GUILDBANKFRAME_OPENED" then
    Guild:OnGuildBankOpened()
  elseif event == "GUILDBANKFRAME_CLOSED" then
    Guild:OnGuildBankClosed()
  elseif event == "GUILD_ROSTER_UPDATE" then
    Guild:OnGuildRosterUpdate()
  end
end)

-- ============================================================================
-- Initialization
-- ============================================================================

if NS.Log then
  NS.Log("INFO", "Guild module loaded (v3.0.1 - Fixed Deduplication)")
end

return NS.Guild