-- tracker_deaths.lua (CORRECTED)
-- WhoDAT - Robust Death Tracking for Wrath 3.3.5a (Warmane-safe)
-- Multiple detection methods for accurate killer identification

local ADDON_NAME = "WhoDAT"
local NS = _G[ADDON_NAME] or {}
_G[ADDON_NAME] = NS

-- ============================================================================
-- Unit Filters (adapted from DeathNote) + Warmane-safe flag helpers
-- ============================================================================

local WD_UnitFilters = {}
local bit_band, bit_bor = bit.band, bit.bor

-- Private-core safety: treat unknown CLEU destination as the player
-- Set false to be stricter in raids/instances
local WD_AssumeUnknownDestIsPlayer = true

local WD_FilterSettings = {
  group = true,           -- mine/party/raid players
  my_pet = true,
  friendly_players = true,
  enemy_players = true,
  friendly_npcs = true,
  enemy_npcs = true,
  other_pets = true,
}

local function WD_ToNumberFlags(flags)
  if flags == nil then return 0 end
  local t = type(flags)
  if t == "number" then return flags end
  if t == "string" then
    -- Try decimal first, then hex (e.g., "0x00000008")
    local n = tonumber(flags) or tonumber(flags, 16)
    return n or 0
  end
  return 0
end

local function WD_Band(a, b)
  return bit_band(WD_ToNumberFlags(a), WD_ToNumberFlags(b))
end

local function WD_UpdateUnitFilters()
  wipe(WD_UnitFilters)
  if WD_FilterSettings.group then
    table.insert(WD_UnitFilters, bit_bor(COMBATLOG_OBJECT_AFFILIATION_MINE,  COMBATLOG_OBJECT_TYPE_PLAYER))
    table.insert(WD_UnitFilters, bit_bor(COMBATLOG_OBJECT_AFFILIATION_PARTY, COMBATLOG_OBJECT_TYPE_PLAYER))
    table.insert(WD_UnitFilters, bit_bor(COMBATLOG_OBJECT_AFFILIATION_RAID,  COMBATLOG_OBJECT_TYPE_PLAYER))
  end
  if WD_FilterSettings.my_pet then
    table.insert(WD_UnitFilters, bit_bor(COMBATLOG_OBJECT_AFFILIATION_MINE, COMBATLOG_OBJECT_TYPE_PET))
  end
  if WD_FilterSettings.friendly_players then
    table.insert(WD_UnitFilters, bit_bor(COMBATLOG_OBJECT_TYPE_PLAYER, COMBATLOG_OBJECT_REACTION_FRIENDLY))
  end
  if WD_FilterSettings.enemy_players then
    table.insert(WD_UnitFilters, bit_bor(COMBATLOG_OBJECT_TYPE_PLAYER, COMBATLOG_OBJECT_REACTION_HOSTILE))
  end
  if WD_FilterSettings.friendly_npcs then
    table.insert(WD_UnitFilters, bit_bor(COMBATLOG_OBJECT_TYPE_NPC, COMBATLOG_OBJECT_REACTION_FRIENDLY))
  end
  if WD_FilterSettings.enemy_npcs then
    table.insert(WD_UnitFilters, bit_bor(COMBATLOG_OBJECT_TYPE_NPC, COMBATLOG_OBJECT_REACTION_HOSTILE))
  end
  if WD_FilterSettings.other_pets then
    if WD_FilterSettings.friendly_players then
      table.insert(WD_UnitFilters, bit_bor(COMBATLOG_OBJECT_REACTION_FRIENDLY, COMBATLOG_OBJECT_CONTROL_PLAYER, COMBATLOG_OBJECT_TYPE_PET))
    end
    if WD_FilterSettings.enemy_players then
      table.insert(WD_UnitFilters, bit_bor(COMBATLOG_OBJECT_REACTION_HOSTILE, COMBATLOG_OBJECT_CONTROL_PLAYER, COMBATLOG_OBJECT_TYPE_PET))
    end
    if WD_FilterSettings.friendly_npcs then
      table.insert(WD_UnitFilters, bit_bor(COMBATLOG_OBJECT_REACTION_FRIENDLY, COMBATLOG_OBJECT_CONTROL_NPC,    COMBATLOG_OBJECT_TYPE_PET))
    end
    if WD_FilterSettings.enemy_npcs then
      table.insert(WD_UnitFilters, bit_bor(COMBATLOG_OBJECT_REACTION_HOSTILE, COMBATLOG_OBJECT_CONTROL_NPC,     COMBATLOG_OBJECT_TYPE_PET))
    end
  end
end

local function ObjectFlagsToType(flags)
  local f = WD_ToNumberFlags(flags)
  if CL_TYPE_PLAYER and bit_band(f, CL_TYPE_PLAYER) > 0 then return "player" end
  if CL_TYPE_PET    and bit_band(f, CL_TYPE_PET)    > 0 then return "pet"    end
  if CL_TYPE_NPC    and bit_band(f, CL_TYPE_NPC)    > 0 then return "npc"    end
  return "unknown"
end

local function WD_IsFiltered(sourceFlags, destFlags)
  -- NPC-only fast path for private-core "unknown dest" cases:
  local srcType = ObjectFlagsToType(sourceFlags)
  if srcType == "npc" and WD_ToNumberFlags(destFlags) == 0 then
    return true
  end

  for i = 1, #WD_UnitFilters do
    local f = WD_UnitFilters[i]
    if WD_Band(sourceFlags, f) == f or WD_Band(destFlags, f) == f then
      return true
    end
  end
  return false
end

WD_UpdateUnitFilters()

-- Cache common COMBATLOG constants defensively
local CL_AFFIL_MINE   = _G.COMBATLOG_OBJECT_AFFILIATION_MINE
local CL_TYPE_PLAYER  = _G.COMBATLOG_OBJECT_TYPE_PLAYER
local CL_TYPE_PET     = _G.COMBATLOG_OBJECT_TYPE_PET
local CL_TYPE_NPC     = _G.COMBATLOG_OBJECT_TYPE_NPC

-- ============================================================================
-- Warmane-specific name normalization
-- ============================================================================

-- On Warmane: sourceGUID is actually the *name*; sourceName can be numeric
local function WD_NormalizeSourceName(sourceName, sourceGUID)
  if type(sourceGUID) == "string" and sourceGUID:find("%a") then
    return sourceGUID
  end
  if type(sourceName) == "string" and sourceName ~= "" then
    return sourceName
  end
  if type(sourceName) == "number" then
    if UnitExists("target") then
      local tName = UnitName("target")
      if tName and tName ~= "" then
        return tName
      end
    end
    return ("Creature #%d"):format(sourceName)
  end
  return "Unknown"
end

-- ============================================================================
-- State
-- ============================================================================

local deathState = {
  -- Multi-source attacker tracking
  recentAttackers   = {},   -- [name] -> {name,type,lastHit,totalDamage,hits,spells,...}
  lastDamageEvent   = nil,

  -- Environmental tracking
  activeDebuffs     = {},
  waterBreathing    = 0,

  -- Combat tracking
  combatStartTime   = nil,
  inCombat          = false,

  -- Death tracking
  durabilityBeforeDeath = nil,
  deathTime         = nil,  -- epoch (time())
  deathClock        = nil,  -- uptime (GetTime())
  pendingDeath      = nil,

  -- Killing blow tracking
  killingBlow       = nil,
}

local ATTACKER_TIMEOUT     = 10  -- seconds
local DAMAGE_HISTORY_SIZE  = 20  -- (reserved for future use)

-- ============================================================================
-- Attacker bookkeeping
-- ============================================================================

local function RecordAttacker(sourceGUID, sourceName, sourceFlags, damageAmount, overkill, spellName)
  if not sourceGUID and not sourceName then return end
  local now       = GetTime()
  local normName  = WD_NormalizeSourceName(sourceName, sourceGUID)
  local sourceType = ObjectFlagsToType(sourceFlags)

  -- Warmane quirk: sometimes flags are unreliable for friendly NPCs
  -- Only flip to NPC if it's marked as friendly AND not a party/raid member
  if sourceType == "player" then
    local myName = UnitName("player")
    if normName ~= myName then
      -- Check if it's a party/raid member
      local isGroupMember = false
      if GetNumRaidMembers() > 0 then
        for i = 1, 40 do
          if UnitName("raid"..i) == normName then
            isGroupMember = true
            break
          end
        end
      elseif GetNumPartyMembers() > 0 then
        for i = 1, 4 do
          if UnitName("party"..i) == normName then
            isGroupMember = true
            break
          end
        end
      end
      
      -- Only flip to NPC if it's friendly (not hostile) AND not in our group
      -- This preserves enemy player detection for PvP
      if not isGroupMember then
        local REACTION_FRIENDLY = _G.COMBATLOG_OBJECT_REACTION_FRIENDLY
        if REACTION_FRIENDLY and WD_Band(sourceFlags, REACTION_FRIENDLY) > 0 then
          sourceType = "npc"
        end
        -- Otherwise keep as "player" for enemy players
      end
    end
  end

  local attacker = deathState.recentAttackers[normName]
  if not attacker then
    attacker = {
      guid           = nil,      -- GUIDs unreliable on Warmane
      name           = normName,
      type           = sourceType,
      firstHit       = now,
      lastHit        = now,
      totalDamage    = 0,
      hits           = 0,
      spells         = {},
      raw_sourceGUID = sourceGUID,
      raw_sourceName = sourceName,
    }
    deathState.recentAttackers[normName] = attacker
  end

  attacker.lastHit     = now
  attacker.totalDamage = attacker.totalDamage + (type(damageAmount) == "number" and damageAmount or 0)
  attacker.hits        = attacker.hits + 1
  if spellName then
    attacker.spells[spellName] = (attacker.spells[spellName] or 0) + 1
  end

  if overkill and type(overkill) == "number" and overkill > 0 then
    deathState.killingBlow = {
      attacker  = attacker,
      damage    = damageAmount,
      overkill  = overkill,
      spell     = spellName,
      timestamp = now,
    }
  end
end

-- ============================================================================
-- CLEU handler (relaxed gate for private cores)
-- ============================================================================

local function OnCombatLog(...)
  -- Capture varargs once; avoid nested use of `...`
  local args = {...}

  -- Wrath/Warmane CLEU header (11 values)
  local timestamp     = args[1]
  local subevent      = args[2]
  local hideCaster    = args[3]
  local sourceGUID    = args[4]
  local sourceName    = args[5]
  local sourceFlags   = args[6]
  local sourceRaidFlags = args[7]
  local destGUID      = args[8]
  local destName      = args[9]
  local destFlags     = args[10]
  local destRaidFlags = args[11]

  -- Diagnostic
  if subevent == "UNIT_DIED" then
    NS.Log("DEBUG", "CLEU: UNIT_DIED (destName=%s flags=0x%x)", tostring(destName), WD_ToNumberFlags(destFlags))
    -- do not return
  end

  -- ---- Private-core safe "is me" test with NPC-only unknown dest gate ----
  local srcType     = ObjectFlagsToType(sourceFlags)           -- "player" | "pet" | "npc" | "unknown"
  local destFlagNum = WD_ToNumberFlags(destFlags)
  local mineByFlag  = CL_AFFIL_MINE and WD_Band(destFlags, CL_AFFIL_MINE) > 0
  local destLooksMe = (type(destName) == "string" and destName ~= "" and destName == UnitName("player"))
  local unknownDest = (destName == nil) and (destFlagNum == 0)

  local treatAsMine =
       mineByFlag
    or destLooksMe
    or (unknownDest and WD_AssumeUnknownDestIsPlayer and srcType == "npc")
    or (unknownDest and srcType == "npc" and UnitAffectingCombat and UnitAffectingCombat("player"))

  if not treatAsMine then
    return
  end

  -- Apply filter normally; bypass ONLY for unknown-dest NPC
  local passesFilter = WD_IsFiltered(sourceFlags, destFlags)
  if not passesFilter and not (unknownDest and srcType == "npc") then
    return
  end

  -- ---- Safe readers over args[] (no tonumber(nil); only accept right types) ----
  local function getNum(slot)
    local v = args[slot]
    return (type(v) == "number") and v or nil
  end

  local function getStr(slot)
    local v = args[slot]
    return (type(v) == "string") and v or nil
  end

  -- ---- Extract and record damage (defensive scanning) ----
  local amount, overkill, spellName

  if subevent == "SWING_DAMAGE" then
    -- typical amount at 12; scan forward defensively
    amount   = getNum(12) or getNum(13) or getNum(14) or getNum(15) or getNum(16) or getNum(17) or getNum(18)
    -- overkill: next plausible numeric slot; default 0
    overkill = (amount and (getNum(13) or getNum(14) or 0)) or 0
    spellName = "Melee"

  elseif subevent == "ENVIRONMENTAL_DAMAGE" then
    local environmentalType = getStr(12)
    spellName = environmentalType or "Environmental"
    amount    = getNum(13) or getNum(14) or getNum(15) or getNum(16) or getNum(17) or getNum(18)
    overkill  = (amount and (getNum(14) or getNum(15) or 0)) or 0

  elseif subevent == "SPELL_DAMAGE"
      or subevent == "SPELL_PERIODIC_DAMAGE"
      or subevent == "RANGE_DAMAGE"
      or subevent == "DAMAGE_SHIELD"
      or subevent == "DAMAGE_SPLIT" then
    -- usually spellId@12, spellName@13, spellSchool@14, amount@15, overkill@16
    spellName = getStr(13) or getStr(12) or "Spell"
    amount    = getNum(15) or getNum(16) or getNum(17) or getNum(18) or getNum(19) or getNum(20)
    overkill  = (amount and (getNum(16) or getNum(17) or 0)) or 0
  end

  -- Record attacker (even when amount == 0, to keep roster)
  if amount ~= nil then
    RecordAttacker(sourceGUID, sourceName, sourceFlags, amount, overkill or 0, spellName)
  end

  -- Store last positive damage for Method 3
  if amount and amount > 0 then
    local srcType2 = ObjectFlagsToType(sourceFlags)
    local srcName  = WD_NormalizeSourceName(sourceName, sourceGUID)
    deathState.lastDamageEvent = {
      timestamp  = GetTime(),
      source     = srcName,
      sourceGUID = nil,     -- GUID unreliable on Warmane
      sourceType = srcType2,
      spell      = spellName,
      amount     = amount,
      overkill   = overkill or 0,
    }
  end
end

-- ============================================================================
-- Environmental checks
-- ============================================================================

local function CheckEnvironmentalFactors()
  local factors = {}

  -- Drowning (breath timer)
  if IsSwimming() then
    local breath = GetMirrorTimerInfo(2)
    if breath and breath > 0 then
      table.insert(factors, {
        type = "drowning",
        name = "Drowning",
        severity = "high",
        breath_remaining = breath,
      })
    end
  end

  -- Fatigue (out of bounds)
  local _, _, _, fatigueValue = GetMirrorTimerInfo(1)
  if fatigueValue and fatigueValue > 0 then
    table.insert(factors, {
      type = "fatigue",
      name = "Fatigue",
      severity = "fatal",
    })
  end

  -- Active debuffs
  for i = 1, 40 do
    local name, _, _, _, _, _, _, _, _, _, spellId = UnitDebuff("player", i)
    if not name then break end
    local fatalDebuffs = {
      ["Fatigue"] = true,
      ["Chilled"] = true,
    }
    if fatalDebuffs[name] then
      table.insert(factors, {
        type = "debuff",
        name = name,
        severity = "high",
        spellId = spellId,
      })
    end
    deathState.activeDebuffs[name] = spellId
  end

  return factors
end

-- ============================================================================
-- Killer determination
-- ============================================================================

local function GetMostUsedSpell(spells)
  if not spells then return nil end
  local maxSpell, maxCount = nil, 0
  for spell, count in pairs(spells) do
    if count > maxCount then
      maxSpell, maxCount = spell, count
    end
  end
  return maxSpell
end

local function DetermineMostLikelyKiller()
  local now = GetTime()

  -- Method 1: Killing blow
  if deathState.killingBlow and (now - deathState.killingBlow.timestamp) < 2 then
    return {
      name       = deathState.killingBlow.attacker.name,
      type       = deathState.killingBlow.attacker.type,
      guid       = deathState.killingBlow.attacker.guid,
      method     = "killing_blow",
      confidence = "very_high",
      spell      = deathState.killingBlow.spell,
      damage     = deathState.killingBlow.damage,
      overkill   = deathState.killingBlow.overkill,
    }
  end

  -- Method 2: Environmental (fatal severity)
  local envFactors = CheckEnvironmentalFactors()
  for _, factor in ipairs(envFactors) do
    if factor.severity == "fatal" then
      return {
        name       = factor.name,
        type       = "environmental",
        guid       = nil,
        method     = "environmental",
        confidence = "high",
        details    = factor,
      }
    end
  end

  -- Method 3: Last damage near death (anchored to uptime deathClock)
  do
    local nearWindow = 3
    local deathClock = deathState.deathClock
    if deathClock and deathState.lastDamageEvent
       and math.abs(deathClock - deathState.lastDamageEvent.timestamp) < nearWindow then
      local ev = deathState.lastDamageEvent
      NS.Log("DEBUG", "Method3: last_damage_near_death -> %s (%s)",
             tostring(ev.source or "Unknown"), tostring(ev.sourceType or "unknown"))
      return {
        name       = ev.source or "Unknown",
        type       = ev.sourceType or "unknown",
        guid       = nil,
        method     = "last_damage_near_death",
        confidence = "high",
        spell      = ev.spell,
        damage     = ev.amount,
        overkill   = ev.overkill or 0,
      }
    end
  end

  -- Fallback to your original last_damage heuristic (short window)
  if deathState.lastDamageEvent and (GetTime() - deathState.lastDamageEvent.timestamp) < 2 then
    return {
      name       = deathState.lastDamageEvent.source or "Unknown",
      type       = deathState.lastDamageEvent.sourceType or "unknown",
      guid       = deathState.lastDamageEvent.sourceGUID,
      method     = "last_damage_fallback",
      confidence = "high",
      spell      = deathState.lastDamageEvent.spell,
      damage     = deathState.lastDamageEvent.amount,
      overkill   = deathState.lastDamageEvent.overkill or 0,
    }
  end

  -- Method 4: Weighted analysis
  local bestAttacker, bestScore = nil, 0
  for _, attacker in pairs(deathState.recentAttackers) do
    local timeSinceLastHit = now - (attacker.lastHit or 0)
    if timeSinceLastHit < ATTACKER_TIMEOUT then
      local damageScore  = (attacker.totalDamage or 0) / 100
      local recencyScore = ((ATTACKER_TIMEOUT - timeSinceLastHit) / ATTACKER_TIMEOUT) * 100
      local hitsScore    = math.min((attacker.hits or 0) * 5, 20)
      local totalScore   = (damageScore * 0.4) + (recencyScore * 0.4) + (hitsScore * 0.2)
      if totalScore > bestScore then
        bestScore, bestAttacker = totalScore, attacker
      end
    end
  end

  if bestAttacker then
    return {
      name           = bestAttacker.name,
      type           = bestAttacker.type,
      guid           = bestAttacker.guid,
      method         = "weighted_analysis",
      confidence     = "medium",
      total_damage   = bestAttacker.totalDamage,
      hits           = bestAttacker.hits,
      primary_spell  = GetMostUsedSpell(bestAttacker.spells),
    }
  end

  -- Method 5: Target fallback
  if UnitExists("target") and UnitCanAttack("player", "target") then
    local targetName = UnitName("target")
    if deathState.recentAttackers[targetName] then
      return {
        name       = targetName,
        type       = UnitPlayerControlled("target") and "player" or "npc",
        guid       = nil,
        method     = "target_fallback",
        confidence = "low",
      }
    end
  end

  -- Method 6: Environmental guess
  if #envFactors > 0 then
    return {
      name       = envFactors[1].name,
      type       = "environmental",
      guid       = nil,
      method     = "environmental_guess",
      confidence = "low",
      details    = envFactors[1],
    }
  end

  -- None
  return {
    name       = "Unknown",
    type       = "unknown",
    guid       = nil,
    method     = "none",
    confidence = "none",
  }
end

-- ============================================================================
-- Death detection & event construction
-- ============================================================================

local function GetAverageDurability()
  local total, count = 0, 0
  for slot = 1, 18 do
    local current, maximum = GetInventoryItemDurability(slot)
    if current and maximum and maximum > 0 then
      total = total + (current / maximum)
      count = count + 1
    end
  end
  if count == 0 then return 100 end
  return (total / count) * 100
end

local function GetDurabilityLoss(beforePct, afterPct)
  return (beforePct or 0) - (afterPct or 0)
end

local function OnPlayerDead()
  -- Capture durability and timestamps immediately
  deathState.durabilityBeforeDeath = GetAverageDurability()
  deathState.deathTime  = time()      -- epoch for SavedVariables
  deathState.deathClock = GetTime()   -- uptime for Method 3

  -- Location/context
  local zone    = GetRealZoneText() or GetZoneText() or "Unknown"
  local subzone = GetSubZoneText() or ""
  local x, y    = GetPlayerMapPosition("player")
  if not x or not y then
    SetMapToCurrentZone()
    x, y = GetPlayerMapPosition("player")
    if not x or not y then x, y = 0, 0 end
  end

  local inInstance                   = IsInInstance()
  local instanceName, instanceType, difficulty = GetInstanceInfo()
  local groupSize                    = (GetNumRaidMembers and GetNumRaidMembers() or 0)
  groupSize                          = (groupSize > 0) and groupSize or (GetNumPartyMembers and GetNumPartyMembers() or 0)
  local groupType                    = (GetNumRaidMembers and GetNumRaidMembers() or 0) > 0 and "raid"
                                      or ((GetNumPartyMembers and GetNumPartyMembers() or 0) > 0 and "party" or "solo")
  local combatDuration               = (deathState.combatStartTime and (GetTime() - deathState.combatStartTime)) or nil

  -- Allow CLEU to catch up before choosing killer
  _G.WhoDAT_Util.after(0.25, function()
    local killer = DetermineMostLikelyKiller()

    -- Recent attackers list (defensive)
    local recentAttackersList, now = {}, GetTime()
    for _, attacker in pairs(deathState.recentAttackers) do
      local lastHit = attacker.lastHit or 0
      if (now - lastHit) < 30 then
        local displayName = (type(attacker.name) == "string" and attacker.name ~= "") and attacker.name
                            or tostring(attacker.name or "Unknown")
        table.insert(recentAttackersList, {
          name      = displayName,
          type      = attacker.type,
          damage    = attacker.totalDamage or 0,
          hits      = attacker.hits or 0,
          first_hit = attacker.firstHit or 0,
          last_hit  = lastHit,
          spells    = attacker.spells or {},
        })
      end
    end
    if #recentAttackersList > 1 then
      table.sort(recentAttackersList, function(a, b)
        return (a.damage or 0) > (b.damage or 0)
      end)
    end

    local killerName = (type(killer.name) == "string" and killer.name ~= "") and killer.name
                       or tostring(killer.name or "Unknown")

    local deathEvent = {
      ts                = deathState.deathTime,
      -- Location
      zone              = zone,
      subzone           = subzone,
      x = x, y = y,
      level             = UnitLevel("player"),
      -- Killer
      killer_name       = killerName,
      killer_type       = killer.type,
      killer_guid       = nil,
      killer_method     = killer.method,
      killer_confidence = killer.confidence,
      killer_spell      = killer.spell,
      killer_damage     = killer.damage,
      -- Killing blow detail
      killing_blow      = deathState.killingBlow and {
        attacker = tostring(deathState.killingBlow.attacker.name or "Unknown"),
        spell    = deathState.killingBlow.spell,
        damage   = deathState.killingBlow.damage,
        overkill = deathState.killingBlow.overkill,
      } or nil,
      -- Attackers
      recent_attackers  = recentAttackersList,
      attacker_count    = #recentAttackersList,
      -- Instance & group
      in_instance       = inInstance,
      instance_name     = instanceName,
      instance_type     = instanceType,
      instance_difficulty = difficulty,
      group_size        = groupSize,
      group_type        = groupType,
      -- Combat
      combat_duration   = combatDuration,
      in_combat_at_death= deathState.inCombat,
      -- Durability / debuffs
      durability_before = deathState.durabilityBeforeDeath,
      active_debuffs    = deathState.activeDebuffs,
      -- Filled on resurrection
      durability_after  = nil,
      durability_loss   = nil,
      rez_type          = nil,
      rez_time          = nil,
    }

    deathState.pendingDeath = deathEvent

    -- Killer announcement (WARN so it shows at default threshold)
    NS.Log("WARN", "%s killed by %s (%s) in %s%s",
      UnitName("player"), killerName, killer.confidence or "unknown",
      zone, (subzone ~= "" and (" - " .. subzone) or "")
    )

    if #recentAttackersList > 1 then
      NS.Log("DEBUG", "Recent attackers:")
      for i = 1, math.min(5, #recentAttackersList) do
        local a = recentAttackersList[i]
        NS.Log("DEBUG", string.format(" %d. %s (%s damage, %d hits)", i, a.name, a.damage, a.hits))
      end
    end
  end)
end

local function OnPlayerAlive()
  if not deathState.pendingDeath then return end

  local deathDuration = time() - (deathState.deathTime or time())
  local durabilityAfter = GetAverageDurability()
  local durabilityLoss  = GetDurabilityLoss(deathState.durabilityBeforeDeath or 100, durabilityAfter)

  local rezType = "spirit"
  for i = 1, 40 do
    local name = UnitBuff("player", i)
    if not name then break end
    if name == "Soulstone Resurrection" then
      rezType = "soulstone"; break
    elseif name:match("Rebirth") or name:match("Resurrection") or name:match("Redemption") then
      rezType = "class_rez"; break
    end
  end
  if rezType == "spirit" and durabilityLoss < 1 then
    rezType = "class_rez"
  elseif rezType == "spirit" and deathDuration < 10 then
    rezType = "corpse"
  end

  -- Complete & persist
  deathState.pendingDeath.durability_after = durabilityAfter
  deathState.pendingDeath.durability_loss  = durabilityLoss
  deathState.pendingDeath.rez_type         = rezType
  deathState.pendingDeath.rez_time         = deathDuration
  deathState.pendingDeath.rez_ts           = time()

  local key = NS.Utils and NS.Utils.GetPlayerKey and NS.Utils.GetPlayerKey()
              or (GetRealmName() .. ":" .. UnitName("player"))

  WhoDatDB = WhoDatDB or {}
  WhoDatDB.characters = WhoDatDB.characters or {}
  WhoDatDB.characters[key] = WhoDatDB.characters[key] or {}
  WhoDatDB.characters[key].events = WhoDatDB.characters[key].events or {}
  WhoDatDB.characters[key].events.deaths = WhoDatDB.characters[key].events.deaths or {}

  table.insert(WhoDatDB.characters[key].events.deaths, deathState.pendingDeath)

  if NS.EventBus and NS.EventBus.Emit then
    NS.EventBus:Emit("player", "death", deathState.pendingDeath)
  end

  NS.Log("INFO", "Death logged: %s (method: %s, confidence: %s, dur loss: %.1f%%, rez: %s)",
    deathState.pendingDeath.killer_name,
    deathState.pendingDeath.killer_method,
    deathState.pendingDeath.killer_confidence,
    durabilityLoss,
    rezType
  )

  -- Clear state
  deathState.pendingDeath    = nil
  deathState.combatStartTime = nil
  deathState.recentAttackers = {}
  deathState.killingBlow     = nil
  deathState.lastDamageEvent = nil
  deathState.activeDebuffs   = {}
end

-- ============================================================================
-- Maintenance
-- ============================================================================

local function CleanupOldAttackers()
  local now = GetTime()
  for name, attacker in pairs(deathState.recentAttackers) do
    if (now - (attacker.lastHit or 0)) > ATTACKER_TIMEOUT then
      deathState.recentAttackers[name] = nil
    end
  end
end

-- ============================================================================
-- Event Registration
-- ============================================================================

local frame = CreateFrame("Frame")
frame:RegisterEvent("PLAYER_DEAD")
frame:RegisterEvent("PLAYER_ALIVE")
frame:RegisterEvent("PLAYER_UNGHOST")
frame:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED")
frame:RegisterEvent("PLAYER_REGEN_DISABLED")
frame:RegisterEvent("PLAYER_REGEN_ENABLED")
frame:RegisterEvent("MIRROR_TIMER_START") -- Breath/fatigue timers

local timeSinceLastCleanup = 0
frame:SetScript("OnUpdate", function(_, elapsed)
  timeSinceLastCleanup = timeSinceLastCleanup + elapsed
  if timeSinceLastCleanup >= 5 then
    CleanupOldAttackers()
    timeSinceLastCleanup = 0
  end
end)

frame:SetScript("OnEvent", function(_, event, ...)
  if event == "PLAYER_DEAD" then
    OnPlayerDead()
  elseif event == "PLAYER_ALIVE" or event == "PLAYER_UNGHOST" then
    OnPlayerAlive()
  elseif event == "COMBAT_LOG_EVENT_UNFILTERED" then
    OnCombatLog(...)
  elseif event == "PLAYER_REGEN_DISABLED" then
    deathState.combatStartTime = GetTime()
    deathState.inCombat = true
  elseif event == "PLAYER_REGEN_ENABLED" then
    deathState.inCombat = false
    if not deathState.pendingDeath then
      deathState.combatStartTime = nil
    end
  elseif event == "MIRROR_TIMER_START" then
    local timerType = select(1, ...)
    if timerType == "BREATH" then
      deathState.waterBreathing = GetMirrorTimerInfo(2)
    end
  end
end)

-- ============================================================================
-- Debug Commands
-- ============================================================================

SLASH_WDDEATHS1 = "/wddeaths"
SlashCmdList["WDDEATHS"] = function(msg)
  msg = (msg or ""):lower()
  if msg == "stats" then
    local key = NS.Utils and NS.Utils.GetPlayerKey and NS.Utils.GetPlayerKey()
              or (GetRealmName() .. ":" .. UnitName("player"))
    local char   = WhoDatDB and WhoDatDB.characters and WhoDatDB.characters[key]
    local deaths = char and char.events and char.events.deaths or {}

    print("=== Death Statistics ===")
    print(string.format("Total deaths: %d", #deaths))
    if #deaths > 0 then
      local byConfidence, byMethod = {}, {}
      for _, death in ipairs(deaths) do
        local conf   = death.killer_confidence or "unknown"
        local method = death.killer_method     or "unknown"
        byConfidence[conf] = (byConfidence[conf] or 0) + 1
        byMethod[method]   = (byMethod[method]   or 0) + 1
      end
      print("\nKiller Detection Confidence:")
      for conf, count in pairs(byConfidence) do
        print(string.format(" %s: %d (%.1f%%)", conf, count, count / #deaths * 100))
      end
      print("\nDetection Methods Used:")
      for method, count in pairs(byMethod) do
        print(string.format(" %s: %d", method, count))
      end
    end

  elseif msg == "recent" then
    local key = NS.Utils and NS.Utils.GetPlayerKey and NS.Utils.GetPlayerKey()
              or (GetRealmName() .. ":" .. UnitName("player"))
    local char   = WhoDatDB and WhoDatDB.characters and WhoDatDB.characters[key]
    local deaths = char and char.events and char.events.deaths or {}

    print("=== Recent Deaths ===")
    local start = math.max(1, #deaths - 9)
    for i = start, #deaths do
      local death = deaths[i]
      local timestamp = date("%Y-%m-%d %H:%M", death.ts)
      print(string.format("[%s] %s (%s conf, %s) in %s",
        timestamp,
        death.killer_name,
        death.killer_confidence or "?",
        death.killer_method     or "?",
        death.zone))
      if death.attacker_count and death.attacker_count > 1 then
        print(string.format(" + %d other attackers involved", death.attacker_count - 1))
      end
    end

  elseif msg == "attackers" then
    print("=== Current Attackers ===")
    local now = GetTime()
    local count = 0
    for _, attacker in pairs(deathState.recentAttackers) do
      local timeSince = now - (attacker.lastHit or 0)
      print(string.format("%s (%s): %d damage, %d hits, %.1fs ago",
        attacker.name, attacker.type, attacker.totalDamage, attacker.hits, timeSince))
      count = count + 1
    end
    if count == 0 then
      print("No recent attackers")
    end

  else
    print("=== WhoDAT Enhanced Death Tracker ===")
    print("/wddeaths stats   - Show death statistics")
    print("/wddeaths recent  - Show recent deaths")
    print("/wddeaths attackers - Show current attackers")
  end
end

return NS