-- tracker_combat.lua
-- WhoDAT - Combat Metrics Tracking (Wrath 3.3.5a + Warmane-safe)
-- Integrates with Details!, Recount, and Skada when available
-- Falls back to built-in CLEU tracking when no external addon is present

local ADDON_NAME = "WhoDAT"
local NS = _G[ADDON_NAME] or {}
_G[ADDON_NAME] = NS

-- ============================================================================
-- Configuration
-- ============================================================================
local CONFIG = {
  SAMPLE_INTERVAL = 15,   -- Sample combat metrics every 15 seconds
  MIN_COMBAT_DURATION = 2,-- Only log combats longer than 2 seconds
  ENABLE_TRACKING = true, -- Master toggle
  
  -- Set to true to ALWAYS use built-in tracking, even if Details/Recount/Skada are present
  -- Useful for testing, debugging, or if you don't trust external addons
  FORCE_BUILTIN_TRACKING = false,
  
  -- Preferred order for external addons (first detected = first used)
  ADDON_PRIORITY = { "Details", "Skada", "Recount" },
}

-- ============================================================================
-- External Addon Detection & Integration
-- ============================================================================
local activeCombatSource = nil
local externalAddonsAvailable = {}

-- Detection functions for each addon
local COMBAT_ADDON_DETECTORS = {
  Details = function()
    return _G.Details ~= nil
  end,
  
  Recount = function()
    return _G.Recount ~= nil
  end,
  
  Skada = function()
    return _G.Skada ~= nil
  end,
}

-- Detect which combat addons are available
local function DetectCombatAddons()
  externalAddonsAvailable = {}
  
  if CONFIG.FORCE_BUILTIN_TRACKING then
    if NS.Log then
      NS.Log("INFO", "FORCE_BUILTIN_TRACKING enabled - using WhoDAT combat tracking")
    end
    activeCombatSource = "WhoDAT_builtin"
    return
  end
  
  -- Check each addon in priority order
  for _, addonName in ipairs(CONFIG.ADDON_PRIORITY) do
    local detector = COMBAT_ADDON_DETECTORS[addonName]
    if detector and detector() then
      table.insert(externalAddonsAvailable, addonName)
      if not activeCombatSource then
        activeCombatSource = addonName
        if NS.Log then
          NS.Log("INFO", "Using %s for combat metrics", addonName)
        end
      end
    end
  end
  
  -- Fallback to built-in if none found
  if not activeCombatSource then
    activeCombatSource = "WhoDAT_builtin"
    if NS.Log then
      NS.Log("INFO", "No combat addon detected - using built-in tracking")
    end
  end
end

-- ============================================================================
-- Data Extraction Adapters for External Addons
-- ============================================================================

-- Adapter for Details!
-- Details! API: https://github.com/Tercioo/Details-Damage-Meter
local function GetDetailsData()
  if not _G.Details then return nil end
  
  local playerName = UnitName("player")
  
  -- Details.tabela_vigente is the current/active combat
  -- Details.tabela_historico contains historical combats
  local currentCombat = Details.tabela_vigente
  if not currentCombat then return nil end
  
  -- Combat containers: [1]=Damage, [2]=Healing, [3]=Utility, [4]=DamageTaken
  local container_damage = currentCombat[1]
  local container_healing = currentCombat[2]
  local container_damagetaken = currentCombat[4]
  
  if not container_damage then return nil end
  
  -- Get player actor data from each container
  local playerDamage = container_damage:GetActor(playerName)
  local playerHealing = container_healing and container_healing:GetActor(playerName)
  local playerDamageTaken = container_damagetaken and container_damagetaken:GetActor(playerName)
  
  -- Get combat duration (in seconds)
  local duration = currentCombat:GetCombatTime() or 0
  
  -- Build standardized data structure
  local data = {
    total_damage = playerDamage and playerDamage.total or 0,
    total_healing = playerHealing and playerHealing.total or 0,
    total_overheal = playerHealing and playerHealing.totalover or 0,
    total_damage_taken = playerDamageTaken and playerDamageTaken.total or 0,
    duration = duration,
  }
  
  -- Calculate rates
  if duration > 0 then
    data.dps = data.total_damage / duration
    data.hps = data.total_healing / duration
    data.dtps = data.total_damage_taken / duration
  else
    data.dps = 0
    data.hps = 0
    data.dtps = 0
  end
  
  -- Calculate overheal percentage
  local totalHealingDone = data.total_healing + data.total_overheal
  data.overheal_pct = totalHealingDone > 0 
    and (data.total_overheal / totalHealingDone * 100) 
    or 0
  
  return data
end

-- Adapter for Recount
-- Recount stores data in Recount.db2.combatants
local function GetRecountData()
  if not _G.Recount then return nil end
  
  local playerName = UnitName("player")
  
  -- Recount.db2.combatants contains all player/mob data
  local combatants = Recount.db2 and Recount.db2.combatants
  if not combatants then return nil end
  
  local playerData = combatants[playerName]
  if not playerData then return nil end
  
  -- Get combat duration (FightingTime is active combat time)
  local duration = (Recount.db2.FightingTime or 0)
  
  -- Build standardized data structure
  local data = {
    total_damage = playerData.Damage or 0,
    total_healing = playerData.Healing or 0,
    total_overheal = playerData.Overhealing or 0,
    total_damage_taken = playerData.DamageTaken or 0,
    duration = duration,
  }
  
  -- Calculate rates
  if duration > 0 then
    data.dps = data.total_damage / duration
    data.hps = data.total_healing / duration
    data.dtps = data.total_damage_taken / duration
  else
    data.dps = 0
    data.hps = 0
    data.dtps = 0
  end
  
  -- Calculate overheal percentage
  local totalHealingDone = data.total_healing + data.total_overheal
  data.overheal_pct = totalHealingDone > 0 
    and (data.total_overheal / totalHealingDone * 100) 
    or 0
  
  return data
end

-- Adapter for Skada
-- Skada API: https://github.com/Skada-Damage-Meter/Skada
local function GetSkadaData()
  if not _G.Skada then return nil end
  
  -- Get current combat segment
  -- Skada:GetSet("current") returns the active combat
  local current = Skada:GetSet("current")
  if not current then return nil end
  
  local playerName = UnitName("player")
  
  -- Find player in the current set
  local playerData = nil
  if current.players then
    for _, player in pairs(current.players) do
      if player.name == playerName then
        playerData = player
        break
      end
    end
  end
  
  if not playerData then return nil end
  
  -- Get combat duration
  local duration = current.time or 0
  
  -- Build standardized data structure
  local data = {
    total_damage = playerData.damage or 0,
    total_healing = playerData.healing or 0,
    total_overheal = playerData.overheal or 0,
    total_damage_taken = playerData.damagetaken or 0,
    duration = duration,
  }
  
  -- Calculate rates
  if duration > 0 then
    data.dps = data.total_damage / duration
    data.hps = data.total_healing / duration
    data.dtps = data.total_damage_taken / duration
  else
    data.dps = 0
    data.hps = 0
    data.dtps = 0
  end
  
  -- Calculate overheal percentage
  local totalHealingDone = data.total_healing + data.total_overheal
  data.overheal_pct = totalHealingDone > 0 
    and (data.total_overheal / totalHealingDone * 100) 
    or 0
  
  return data
end

-- Generic function to get data from any available addon
local function GetExternalCombatData()
  if not activeCombatSource or activeCombatSource == "WhoDAT_builtin" then
    return nil
  end
  
  if activeCombatSource == "Details" then
    return GetDetailsData()
  elseif activeCombatSource == "Recount" then
    return GetRecountData()
  elseif activeCombatSource == "Skada" then
    return GetSkadaData()
  end
  
  return nil
end

-- ============================================================================
-- GUID -> Level cache (populated from target/mouseover/focus)
-- ============================================================================
local guidLevelCache = {}

local function CacheUnitLevel(unitId)
  if UnitExists(unitId) then
    local guid = UnitGUID(unitId)
    local lvl  = UnitLevel(unitId)
    if guid and lvl and lvl > 0 then
      guidLevelCache[guid] = lvl
    end
  end
end

local function WD_LooksLikeGUID(s)
  return type(s) == "string" and s:find("%-") ~= nil and s:find("%u") ~= nil
end

-- Robust target level resolution (tries target, mouseover, focus, GUID cache; clamps to 0)
local function GetBestTargetLevel(targetName, targetGUID)
  -- Try explicit target if it matches by GUID or name
  if UnitExists("target") then
    local tGUID = UnitGUID("target")
    local tName = UnitName("target")
    if (targetGUID and tGUID == targetGUID) or (targetName and tName == targetName) then
      local lvl = UnitLevel("target")
      if lvl and lvl > 0 then return lvl end
    end
  end

  -- Try mouseover if it matches by name
  if UnitExists("mouseover") then
    local moName = UnitName("mouseover")
    if targetName and moName == targetName then
      local lvl = UnitLevel("mouseover")
      if lvl and lvl > 0 then return lvl end
    end
  end

  -- Try focus if it matches by name
  if UnitExists("focus") then
    local fName = UnitName("focus")
    if targetName and fName == targetName then
      local lvl = UnitLevel("focus")
      if lvl and lvl > 0 then return lvl end
    end
  end

  -- Try GUID cache last
  if targetGUID and guidLevelCache[targetGUID] and guidLevelCache[targetGUID] > 0 then
    return guidLevelCache[targetGUID]
  end

  -- Unknown
  return 0
end

-- ============================================================================
-- Combat State (used for context tracking regardless of data source)
-- ============================================================================
local combatState = {
  active = false,
  startTime = nil,
  startTimeUnix = nil,
  endTime = nil,
  
  -- Built-in tracking (only used if no external addon)
  totalDamage = 0,
  damageByTarget = {},
  totalHealing = 0,
  totalOverhealing = 0,
  totalDamageTaken = 0,
  damageFromSource = {},
  resourceWasted = 0,
  resourceSpent = 0,
  
  -- Context tracking (ALWAYS used - this is what we add value with)
  currentTarget = nil,
  targetLevel   = 0,
  bossEncounter = false,
  instance = nil,
  group    = nil,
  
  -- Sample history (for charting)
  samples = {},
  lastSampleTime = 0,
}

-- ============================================================================
-- Helper Functions
-- ============================================================================
local function GetCurrentTarget()
  if UnitExists("target") then
    local name = UnitName("target")
    local guid = UnitGUID("target")
    local classification = UnitClassification("target")
    
    -- Use enhanced boss detection from boss_detection module
    local isBoss, bossMethod = false, "not_boss"
    if NS.IsBoss then
      isBoss, bossMethod = NS.IsBoss("target", guid, name)
    else
      -- Fallback to old method if boss_detection module not loaded
      isBoss = (classification == "worldboss" or classification == "elite")
      bossMethod = "legacy_detection"
    end
    
    return {
      name = name,
      guid = guid,
      level = math.max(0, UnitLevel("target") or 0),
      classification = classification,
      is_boss = isBoss,
      boss_detection_method = bossMethod,
    }
  end
  return nil
end

local function GetInstanceContext()
  local inInstance, instanceType = IsInInstance()
  if not inInstance then return nil end
  local name, type, difficulty, difficultyName = GetInstanceInfo()
  return {
    name = name,
    type = type,
    difficulty = difficulty,
    difficultyName = difficultyName,
  }
end

local function GetGroupContext()
  -- Wrath 3.3.5a uses GetNumRaidMembers() and GetNumPartyMembers()
  local numRaid  = GetNumRaidMembers()
  local numParty = GetNumPartyMembers()
  if numRaid > 0 then
    return { type = "raid",  size = numRaid }
  elseif numParty > 0 then
    return { type = "party", size = numParty + 1 } -- +1 to include player
  end
  return { type = "solo", size = 1 }
end

-- Opportunistically update combatState.targetLevel from cache and live units
local function UpdateTargetLevelFromCache()
  if not combatState.currentTarget then return end
  local name = combatState.currentTarget.name
  local guid = combatState.currentTarget.guid
  local lvl  = GetBestTargetLevel(name, guid)
  if lvl and lvl > 0 then
    combatState.targetLevel = lvl
  end
end

-- ============================================================================
-- Sample Tracking (for time-series data / charting)
-- ============================================================================
local function SampleCombatMetrics()
  if not combatState.active then return end
  
  local now = GetTime()
  if now - combatState.lastSampleTime < CONFIG.SAMPLE_INTERVAL then
    return
  end
  
  combatState.lastSampleTime = now
  local elapsed = now - combatState.startTime
  
  -- Try to get current metrics from external addon
  local externalData = GetExternalCombatData()
  
  local sample
  if externalData then
    -- Use external addon's data
    sample = {
      timestamp = elapsed,
      dps = externalData.dps or 0,
      hps = externalData.hps or 0,
      dtps = externalData.dtps or 0,
      total_damage = externalData.total_damage or 0,
      total_healing = externalData.total_healing or 0,
      total_damage_taken = externalData.total_damage_taken or 0,
    }
  else
    -- Use built-in data
    sample = {
      timestamp = elapsed,
      dps = elapsed > 0 and (combatState.totalDamage / elapsed) or 0,
      hps = elapsed > 0 and (combatState.totalHealing / elapsed) or 0,
      dtps = elapsed > 0 and (combatState.totalDamageTaken / elapsed) or 0,
      total_damage = combatState.totalDamage,
      total_healing = combatState.totalHealing,
      total_damage_taken = combatState.totalDamageTaken,
    }
  end
  
  table.insert(combatState.samples, sample)
end

-- ============================================================================
-- Built-in Combat Log Parsing (fallback when no external addon present)
-- ============================================================================
-- NOTE: In Wrath, COMBAT_LOG_EVENT_UNFILTERED handler receives varargs:
-- 1: timestamp, 2: subevent, 3: hideCaster, 4: sourceGUID, 5: sourceName, 6: sourceFlags,
-- 7: sourceRaidFlags, 8: destGUID, 9: destName, 10: destFlags, 11: destRaidFlags, then event-specific args.

local function OnCombatLogEvent(...)
  -- Only parse CLEU if we're using built-in tracking
  if activeCombatSource ~= "WhoDAT_builtin" then return end
  if not combatState.active then return end

  -- Cache levels from live units each tick (keeps GUID cache fresh)
  CacheUnitLevel("target")
  CacheUnitLevel("mouseover")
  CacheUnitLevel("focus")
  UpdateTargetLevelFromCache()

  local timestamp, subevent, hideCaster,
        sourceGUID, sourceName, sourceFlags, sourceRaidFlags,
        destGUID, destName, destFlags, destRaidFlags = ...

  local playerGUID = UnitGUID("player")
  local playerName = UnitName("player")
  
  -- WARMANE FIX: Combat log returns player NAME as sourceGUID, not GUID!
  local isPlayerSource = (sourceGUID == playerGUID) or (sourceGUID == playerName) or (sourceName == playerName)
  local isPlayerDest = (destGUID == playerGUID) or (destGUID == playerName) or (destName == playerName)

  -- Damage dealt by player
  if isPlayerSource then
    if subevent == "SWING_DAMAGE" then
      local amount = select(12, ...)
      if type(amount) == "number" then
        combatState.totalDamage = combatState.totalDamage + amount
        if destGUID then
          combatState.damageByTarget[destGUID] = (combatState.damageByTarget[destGUID] or 0) + amount
        end
      end
    elseif subevent == "SPELL_DAMAGE" or subevent == "SPELL_PERIODIC_DAMAGE" or subevent == "RANGE_DAMAGE" then
      -- WARMANE FIX: Damage is at offset 12, not 15!
      local amount = select(15, ...)
      if type(amount) == "number" then
        combatState.totalDamage = combatState.totalDamage + amount
        if destGUID then
          combatState.damageByTarget[destGUID] = (combatState.damageByTarget[destGUID] or 0) + amount
        end
      end
    elseif subevent == "SPELL_HEAL" or subevent == "SPELL_PERIODIC_HEAL" then
      local amount = select(15, ...)
      local overheal = select(16, ...)
      if type(amount) == "number" then
        combatState.totalHealing = combatState.totalHealing + amount
      end
      if type(overheal) == "number" then
        combatState.totalOverhealing = combatState.totalOverhealing + overheal
      end
    end
  end

  -- Damage taken by player
  if isPlayerDest then
    if subevent == "SWING_DAMAGE" then
      local amount = select(12, ...)
      if type(amount) == "number" then
        combatState.totalDamageTaken = combatState.totalDamageTaken + amount
        if sourceGUID then
          combatState.damageFromSource[sourceGUID] = (combatState.damageFromSource[sourceGUID] or 0) + amount
        end
      end
    elseif subevent == "SPELL_DAMAGE" or subevent == "SPELL_PERIODIC_DAMAGE" or subevent == "RANGE_DAMAGE" or subevent == "ENVIRONMENTAL_DAMAGE" then
      local amount = select(15, ...)
      if type(amount) == "number" then
        combatState.totalDamageTaken = combatState.totalDamageTaken + amount
        if sourceGUID then
          combatState.damageFromSource[sourceGUID] = (combatState.damageFromSource[sourceGUID] or 0) + amount
        end
      end
    end
  end
end

-- ============================================================================
-- Combat Start/End Handlers
-- ============================================================================
local function ResetCombatState()
  combatState.totalDamage = 0
  combatState.damageByTarget = {}
  combatState.totalHealing = 0
  combatState.totalOverhealing = 0
  combatState.totalDamageTaken = 0
  combatState.damageFromSource = {}
  combatState.resourceWasted = 0
  combatState.resourceSpent = 0
  combatState.samples = {}
  combatState.lastSampleTime = 0
  combatState.currentTarget = nil
  combatState.targetLevel = 0
  combatState.bossEncounter = false
end

local function OnCombatStart()
  if not CONFIG.ENABLE_TRACKING then return end
  
  combatState.active = true
  combatState.startTime = GetTime()
  combatState.startTimeUnix = time()
  
  ResetCombatState()
  
  -- Capture initial context
  combatState.currentTarget = GetCurrentTarget()
  combatState.instance = GetInstanceContext()
  combatState.group = GetGroupContext()
  
  -- Determine if this is a boss encounter
  if combatState.currentTarget then
    combatState.bossEncounter = combatState.currentTarget.is_boss
    combatState.targetLevel = combatState.currentTarget.level or 0
  end
  
  -- Cache initial target level
  CacheUnitLevel("target")
  UpdateTargetLevelFromCache()
  
  if NS.Log then
    local targetInfo = combatState.currentTarget 
      and string.format("%s (lvl %d, boss=%s)", 
          combatState.currentTarget.name, 
          combatState.targetLevel,
          tostring(combatState.bossEncounter))
      or "none"
    NS.Log("DEBUG", "Combat started - Target: %s, Source: %s", targetInfo, activeCombatSource)
  end
end

local function OnCombatEnd()
  if not combatState.active then return end
  
  combatState.active = false
  combatState.endTime = GetTime()
  local duration = combatState.endTime - combatState.startTime
  
  -- Ignore very short combats
  if duration < CONFIG.MIN_COMBAT_DURATION then
    if NS.Log then NS.Log("DEBUG", "Combat too short (%.1fs), not logging", duration) end
    return
  end

  -- Final cache pass + refresh target level
  CacheUnitLevel("target")
  CacheUnitLevel("mouseover")
  CacheUnitLevel("focus")
  UpdateTargetLevelFromCache()

  -- Take final sample
  SampleCombatMetrics()

  -- Try to get data from external addon first
  local externalData = GetExternalCombatData()
  
  local combatEvent
  
  if externalData then
    -- ========================================================================
    -- USE EXTERNAL ADDON DATA (Details/Recount/Skada)
    -- ========================================================================
    combatEvent = {
      ts = combatState.startTimeUnix,
      duration = externalData.duration,
      
      -- Combat metrics from external addon (more accurate)
      dps = externalData.dps,
      hps = externalData.hps,
      dtps = externalData.dtps,
      overheal_pct = externalData.overheal_pct,
      total_damage = externalData.total_damage,
      total_healing = externalData.total_healing,
      total_overheal = externalData.total_overheal,
      total_damage_taken = externalData.total_damage_taken,
      
      -- Context from WhoDAT tracking (our value-add)
      target = combatState.currentTarget and combatState.currentTarget.name or nil,
      target_level = combatState.targetLevel or 0,
      is_boss = combatState.bossEncounter,
      instance = combatState.instance and combatState.instance.name or nil,
      instance_difficulty = combatState.instance and combatState.instance.difficultyName or nil,
      group_type = combatState.group.type,
      group_size = combatState.group.size,
      zone = GetRealZoneText() or GetZoneText(),
      subzone = GetSubZoneText(),
      
      -- Metadata
      data_source = activeCombatSource,
      samples = combatState.samples,
    }
    
  else
    -- ========================================================================
    -- FALLBACK TO BUILT-IN TRACKING
    -- ========================================================================
    local dps = duration > 0 and (combatState.totalDamage / duration) or 0
    local hps = duration > 0 and (combatState.totalHealing / duration) or 0
    local dtps = duration > 0 and (combatState.totalDamageTaken / duration) or 0
    local overhealPct = 0
    if (combatState.totalHealing + combatState.totalOverhealing) > 0 then
      overhealPct = (combatState.totalOverhealing / (combatState.totalHealing + combatState.totalOverhealing)) * 100
    end

    combatEvent = {
      ts = combatState.startTimeUnix,
      duration = duration,
      
      -- Built-in metrics
      dps = dps,
      hps = hps,
      dtps = dtps,
      overheal_pct = overhealPct,
      total_damage = combatState.totalDamage,
      total_healing = combatState.totalHealing,
      total_overheal = combatState.totalOverhealing,
      total_damage_taken = combatState.totalDamageTaken,
      
      -- Context
      target = combatState.currentTarget and combatState.currentTarget.name or nil,
      target_level = combatState.targetLevel or 0,
      is_boss = combatState.bossEncounter,
      instance = combatState.instance and combatState.instance.name or nil,
      instance_difficulty = combatState.instance and combatState.instance.difficultyName or nil,
      group_type = combatState.group.type,
      group_size = combatState.group.size,
      zone = GetRealZoneText() or GetZoneText(),
      subzone = GetSubZoneText(),
      
      -- Metadata
      data_source = "WhoDAT_builtin",
      samples = combatState.samples,
    }
  end

  -- Save to database
  local key = NS.Utils and NS.Utils.GetPlayerKey and NS.Utils.GetPlayerKey()
    or (GetRealmName() .. ":" .. UnitName("player"))
  WhoDatDB = WhoDatDB or {}
  WhoDatDB.characters = WhoDatDB.characters or {}
  WhoDatDB.characters[key] = WhoDatDB.characters[key] or {}
  WhoDatDB.characters[key].events = WhoDatDB.characters[key].events or {}
  WhoDatDB.characters[key].events.combat = WhoDatDB.characters[key].events.combat or {}
  table.insert(WhoDatDB.characters[key].events.combat, combatEvent)

  -- Limit storage (keep last 500 combats)
  local maxCombats = 500
  while #WhoDatDB.characters[key].events.combat > maxCombats do
    table.remove(WhoDatDB.characters[key].events.combat, 1)
  end

  -- Emit to EventBus
  if NS.EventBus and NS.EventBus.Emit then
    NS.EventBus:Emit("combat", "ended", combatEvent)
  end

  if NS.Log then
    NS.Log("INFO", "Combat ended: %.0fs, DPS=%.0f HPS=%.0f DTPS=%.0f (source=%s, lvl=%d)",
      duration, combatEvent.dps, combatEvent.hps, combatEvent.dtps, combatEvent.data_source, combatEvent.target_level or 0)
  end
end

-- ============================================================================
-- Event Registration
-- ============================================================================
local frame = CreateFrame("Frame")
frame:RegisterEvent("PLAYER_REGEN_DISABLED")
frame:RegisterEvent("PLAYER_REGEN_ENABLED")
frame:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED")
frame:RegisterEvent("PLAYER_LOGIN")

-- Sample timer + periodic level caching
local sampleTimer = 0
frame:SetScript("OnUpdate", function(self, elapsed)
  sampleTimer = sampleTimer + elapsed

  -- Keep cache warm even if sample interval hasn't elapsed
  if combatState.active then
    CacheUnitLevel("target")
    CacheUnitLevel("mouseover")
    CacheUnitLevel("focus")
    UpdateTargetLevelFromCache()
  end

  if sampleTimer >= CONFIG.SAMPLE_INTERVAL then
    sampleTimer = 0
    SampleCombatMetrics()
  end
end)

frame:SetScript("OnEvent", function(self, event, ...)
  if event == "PLAYER_LOGIN" then
    -- Detect available combat addons on login
    DetectCombatAddons()
    
  elseif event == "PLAYER_REGEN_DISABLED" then
    OnCombatStart()
    
  elseif event == "PLAYER_REGEN_ENABLED" then
    OnCombatEnd()
    
  elseif event == "COMBAT_LOG_EVENT_UNFILTERED" then
    OnCombatLogEvent(...)
  end
end)

-- ============================================================================
-- Public API
-- ============================================================================
function NS.Combat_Enable()  
  CONFIG.ENABLE_TRACKING = true  
end

function NS.Combat_Disable() 
  CONFIG.ENABLE_TRACKING = false 
end

function NS.Combat_SetSampleInterval(seconds)
  CONFIG.SAMPLE_INTERVAL = math.max(5, seconds)
end

function NS.Combat_GetActiveSource()
  return activeCombatSource
end

function NS.Combat_GetAvailableAddons()
  return externalAddonsAvailable
end

function NS.Combat_ForceBuiltinTracking(enable)
  CONFIG.FORCE_BUILTIN_TRACKING = enable
  DetectCombatAddons() -- Re-detect with new setting
end

-- ============================================================================
-- SQL Schema (reference)
-- ============================================================================
--[[ 
CREATE TABLE combat_encounters (
  id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  character_id BIGINT UNSIGNED NOT NULL,
  ts INT UNSIGNED NOT NULL,
  duration FLOAT NOT NULL,
  -- Performance metrics
  dps FLOAT,
  hps FLOAT,
  dtps FLOAT,
  overheal_pct FLOAT,
  -- Totals
  total_damage BIGINT UNSIGNED,
  total_healing BIGINT UNSIGNED,
  total_overheal BIGINT UNSIGNED,
  total_damage_taken BIGINT UNSIGNED,
  -- Context
  target VARCHAR(128),
  target_level TINYINT UNSIGNED,
  is_boss BOOLEAN,
  instance VARCHAR(128),
  instance_difficulty VARCHAR(64),
  group_type ENUM('solo', 'party', 'raid'),
  group_size TINYINT UNSIGNED,
  zone VARCHAR(128),
  subzone VARCHAR(128),
  data_source VARCHAR(32),  -- NEW: tracks where data came from
  KEY idx_char_ts (character_id, ts),
  KEY idx_target (target),
  KEY idx_instance (instance),
  KEY idx_boss (is_boss, dps DESC),
  KEY idx_source (data_source),
  CONSTRAINT fk_combat_char FOREIGN KEY (character_id)
  REFERENCES characters(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
]]

-- ============================================================================
-- Debug Commands
-- ============================================================================
SLASH_WDCOMBAT1 = "/wdcombat"
SlashCmdList["WDCOMBAT"] = function(msg)
  msg = (msg or ""):lower()
  
  if msg == "enable" then
    NS.Combat_Enable()
    print("[WhoDAT] Combat tracking enabled")
    
  elseif msg == "disable" then
    NS.Combat_Disable()
    print("[WhoDAT] Combat tracking disabled")
    
  elseif msg == "source" then
    print("=== Combat Data Source ===")
    print(string.format("Active source: %s", activeCombatSource or "none"))
    print(string.format("Force builtin: %s", tostring(CONFIG.FORCE_BUILTIN_TRACKING)))
    if #externalAddonsAvailable > 0 then
      print("Available addons:")
      for _, addon in ipairs(externalAddonsAvailable) do
        print(string.format("  - %s", addon))
      end
    else
      print("No external combat addons detected")
    end
    
  elseif msg == "forcebuiltin" then
    CONFIG.FORCE_BUILTIN_TRACKING = not CONFIG.FORCE_BUILTIN_TRACKING
    DetectCombatAddons()
    print(string.format("[WhoDAT] Force builtin tracking: %s", tostring(CONFIG.FORCE_BUILTIN_TRACKING)))
    print(string.format("[WhoDAT] Now using: %s", activeCombatSource))
    
  elseif msg == "stats" then
    local key = NS.Utils and NS.Utils.GetPlayerKey and NS.Utils.GetPlayerKey()
      or (GetRealmName() .. ":" .. UnitName("player"))
    local char = WhoDatDB and WhoDatDB.characters and WhoDatDB.characters[key]
    local combats = char and char.events and char.events.combat or {}
    print("=== Combat Statistics ===")
    print(string.format("Total combats: %d", #combats))
    if #combats > 0 then
      local totalDPS, totalHPS, maxDPS, bossFights = 0, 0, 0, 0
      local sourceCounts = {}
      for _, combat in ipairs(combats) do
        totalDPS = totalDPS + (combat.dps or 0)
        totalHPS = totalHPS + (combat.hps or 0)
        maxDPS   = math.max(maxDPS, combat.dps or 0)
        if combat.is_boss then bossFights = bossFights + 1 end
        
        local src = combat.data_source or "unknown"
        sourceCounts[src] = (sourceCounts[src] or 0) + 1
      end
      print(string.format("Average DPS: %.0f", totalDPS / #combats))
      print(string.format("Average HPS: %.0f", totalHPS / #combats))
      print(string.format("Max DPS: %.0f", maxDPS))
      print(string.format("Boss fights: %d", bossFights))
      print("Data sources:")
      for source, count in pairs(sourceCounts) do
        print(string.format("  %s: %d (%.1f%%)", source, count, (count / #combats) * 100))
      end
    end
    
  elseif msg == "current" then
    if combatState.active then
      local elapsed = GetTime() - combatState.startTime
      local externalData = GetExternalCombatData()
      
      print("=== Current Combat ===")
      print(string.format("Duration: %.0fs", elapsed))
      print(string.format("Data source: %s", activeCombatSource))
      
      if externalData then
        print(string.format("DPS: %.0f (external)", externalData.dps))
        print(string.format("HPS: %.0f (external)", externalData.hps))
        print(string.format("Total damage: %d (external)", externalData.total_damage))
      else
        local dps = elapsed > 0 and (combatState.totalDamage / elapsed) or 0
        print(string.format("DPS: %.0f (builtin)", dps))
        print(string.format("Total damage: %d (builtin)", combatState.totalDamage))
      end
      
      print(string.format("Target: %s (lvl %d)",
        combatState.currentTarget and combatState.currentTarget.name or "none",
        combatState.targetLevel or 0))
      print(string.format("Samples: %d", #combatState.samples))
    else
      print("[WhoDAT] Not in combat")
    end
    
  else
    print("=== WhoDAT Combat Tracker ===")
    print("/wdcombat enable       - Enable combat tracking")
    print("/wdcombat disable      - Disable combat tracking")
    print("/wdcombat source       - Show active data source")
    print("/wdcombat forcebuiltin - Toggle force builtin tracking")
    print("/wdcombat stats        - Show combat statistics")
    print("/wdcombat current      - Show current combat info")
  end
end

return NS