--[[========================================================================================
 WhoDAT - tracker_auction.lua (REFACTORED with Auctioneer Integration)
 FULL ORIGINAL FUNCTIONALITY PRESERVED + Auctioneer addon support as data helper
 OUTPUT SCHEMA UNCHANGED - Auctioneer used to improve data quality only
 Auction House scanner + market micro-snapshots + time series + sold/expired tracking
 Integrates with: Auctioneer (Auc-Advanced)
 Wrath 3.3.5a + Warmane compatible
==========================================================================================]]--

local TrackerAuction = {}
TrackerAuction.__index = TrackerAuction
_G.TrackerAuction = TrackerAuction

-- ============================================================================
-- CONFIGURATION (NEW: Added for Auctioneer integration)
-- ============================================================================
local CONFIG = {
  -- Force built-in scanning even if Auctioneer is present
  FORCE_BUILTIN_SCANNING = false,
  
  -- Use Auctioneer for enhanced seller name detection
  USE_AUCTIONEER_FOR_SELLERS = true,
  
  -- Auto-populate market data when viewing auctions
  AUTO_MARKET_SCAN = true,
  
  -- NEW: Piggyback on Auctioneer's scan (best option - real sellers + Auctioneer UI)
  PIGGYBACK_AUCTIONEER_SCAN = false,
  
  -- Prefer Auctioneer instant prices (DEPRECATED - piggyback is better)
  PREFER_AUCTIONEER_PRICES = false,  -- Set to false, use piggyback instead
  
  -- Show confirmation popup before manual AH scan (only if no Auctioneer)
  CONFIRM_MANUAL_SCAN = true,
  
  -- Addon preference order
  ADDON_PRIORITY = { "Auctioneer" },
}
-- ============================================================================
-- EXTERNAL ADDON DETECTION (NEW)
-- ============================================================================
local activeAuctionSource = nil
local externalAddonsAvailable = {}

local AUCTION_ADDON_DETECTORS = {
  Auctioneer = function()
    -- Auctioneer (Auc-Advanced) detection
    return _G.AucAdvanced ~= nil 
      and _G.AucAdvanced.API ~= nil
      and _G.AucAdvanced.Modules ~= nil
  end,
}

local function DetectAuctionAddons()
  externalAddonsAvailable = {}
  
  if CONFIG.FORCE_BUILTIN_SCANNING then
    activeAuctionSource = "WhoDAT_builtin"
    return
  end
  
  -- Check each addon in priority order
  for _, addonName in ipairs(CONFIG.ADDON_PRIORITY) do
    local detector = AUCTION_ADDON_DETECTORS[addonName]
    if detector and detector() then
      table.insert(externalAddonsAvailable, addonName)
      if not activeAuctionSource then
        activeAuctionSource = addonName
      end
    end
  end
  
  -- Fallback to built-in
  if not activeAuctionSource then
    activeAuctionSource = "WhoDAT_builtin"
  end
end
-- NEW: Detect if Auctioneer is currently scanning
local function IsAuctioneerBusy()
  if not _G.AucAdvanced then return false end
  
  -- Try known patterns if present
  local ok, scanning = pcall(function()
    if AucAdvanced.Scan and AucAdvanced.Scan.IsScanning then
      return AucAdvanced.Scan.IsScanning()
    end
    if AucAdvanced.API and AucAdvanced.API.IsScanning then
      return AucAdvanced.API.IsScanning()
    end
    return false
  end)
  
  return ok and scanning
end

-- Trigger Auctioneer to start a scan
local function TriggerAuctioneerScan()
  if not _G.AucAdvanced then return false end
  
  -- Try to start an Auctioneer scan
  local ok, result = pcall(function()
    -- Common scan trigger methods in Auctioneer
    if AucAdvanced.Scan and AucAdvanced.Scan.StartScan then
      AucAdvanced.Scan.StartScan()
      return true
    end
    if AucAdvanced.API and AucAdvanced.API.StartScan then
      AucAdvanced.API.StartScan()
      return true
    end
    -- Alternative: trigger via scan button
    if AucAdvanced.Scan and AucAdvanced.Scan.ScanButton then
      AucAdvanced.Scan.ScanButton:Click()
      return true
    end
    return false
  end)
  
  return ok and result
end
local MARKET_DATA_CACHE = {}

local function CacheMarketData(itemId, stackSize, rows)
  local key = string.format("%d:%d", itemId, stackSize)
  MARKET_DATA_CACHE[key] = {
    rows = rows,
    ts = time(),
  }
end

local function GetCachedMarketData(itemId, stackSize, maxAge)
  maxAge = maxAge or 300  -- 5 minutes default
  local key = string.format("%d:%d", itemId, stackSize)
  local cached = MARKET_DATA_CACHE[key]
  
  if cached and (time() - cached.ts) <= maxAge then
    return cached.rows
  end
  
  return nil
end
-- ============================================================================
-- AUCTIONEER INTEGRATION HELPERS (NEW)
-- These help improve data quality without changing output schema
-- ============================================================================

-- Try to get seller name from Auctioneer's database
-- Used as a fallback when WhoDAT can't extract seller from UI/tooltip
local function GetAuctioneerSeller(itemLink)
  if not CONFIG.USE_AUCTIONEER_FOR_SELLERS then return nil end
  if not _G.AucAdvanced then return nil end
  
  -- Extract itemId from link
  local itemId = itemLink and tonumber(itemLink:match("item:(%d+)"))
  if not itemId then return nil end
  
  -- Try multiple Auctioneer APIs to get seller info
  local ok, seller = pcall(function()
    
    -- Method 1: Check current scan image
    if AucAdvanced.Scan and AucAdvanced.Scan.GetImage then
      local image = AucAdvanced.Scan.GetImage()
      if image and type(image) == "table" then
        -- Scan through current image data
        for i = 1, (image.numrows or 0) do
          if image[i] and image[i].itemId == itemId then
            local owner = image[i].owner or image[i].seller
            if owner and owner ~= "" then
              return owner
            end
          end
        end
      end
    end
    
    -- Method 2: Check Auctioneer's internal cache
    if AucAdvanced.API and AucAdvanced.API.GetAuction then
      local auctions = AucAdvanced.API.GetAuction(itemId)
      if auctions and auctions[1] then
        local owner = auctions[1].owner or auctions[1].seller
        if owner and owner ~= "" then
          return owner
        end
      end
    end
    
    -- Method 3: Try GetImageIterator if available
    if AucAdvanced.Scan and AucAdvanced.Scan.GetImageIterator then
      for entry in AucAdvanced.Scan.GetImageIterator() do
        if entry.itemId == itemId then
          local owner = entry.owner or entry.seller
          if owner and owner ~= "" then
            return owner
          end
        end
      end
    end
    
    return nil
  end)
  
  if ok and seller and seller ~= "" then
    return seller
  end
  
  return nil
end

-- Validate that a price seems reasonable (sanity check)
-- Returns true if price passes sanity checks
local function ValidatePrice(itemLink, price)
  if not price or price <= 0 then return false end
  
  -- Use Auctioneer to validate if available
  if _G.AucAdvanced and _G.AucAdvanced.API then
    local ok, marketValue = pcall(function()
      return _G.AucAdvanced.API.GetMarketValue(itemLink)
    end)
    
    if ok and marketValue and marketValue > 0 then
      -- If price is wildly different from market (>1000x), might be bad data
      local ratio = price / marketValue
      if ratio > 1000 or ratio < 0.001 then
        -- Suspicious price, but don't reject - just flag for review
        return true -- Still accept it, but we know it's odd
      end
    end
  end
  
  return true
end

-- NEW: Get instant market data from Auctioneer (no AH scanning needed)
local function GetInstantMarketData(itemId, stackSize, itemLink)
  if not CONFIG.PREFER_AUCTIONEER_PRICES then return nil end
  if not _G.AucAdvanced or not _G.AucAdvanced.API then return nil end
  
  local api = _G.AucAdvanced.API
  
  -- Get market value from Auctioneer
  local ok, marketValue = pcall(function()
    return api.GetMarketValue(itemLink)
  end)
  
  if not ok or not marketValue or marketValue <= 0 then
    return nil
  end
  
  -- Calculate price per item
  local pricePerItem = math.floor(marketValue / (stackSize > 0 and stackSize or 1) + 0.5)
  
  -- Create market data structure (mimics the scanned structure)
  local marketData = {
    low = {
      {
        priceStack = marketValue,
        priceItem = pricePerItem,
        seller = "Market",
        link = itemLink,
      }
    },
    high = {
      {
        priceStack = marketValue,
        priceItem = pricePerItem,
        seller = "Market",
        link = itemLink,
      }
    }
  }
  
  return marketData
end

-- NEW: Populate market time-series instantly using Auctioneer
local function PopulateMarketDataInstant(myRows)
  -- Skip if piggyback mode is enabled (piggyback provides better data with real sellers)
  if CONFIG.PIGGYBACK_AUCTIONEER_SCAN and activeAuctionSource == "Auctioneer" then
    return 0
  end
  
  if not CONFIG.AUTO_MARKET_SCAN then return 0 end
  if not CONFIG.PREFER_AUCTIONEER_PRICES then return 0 end
  if not _G.AucAdvanced then return 0 end
  
  local count = 0
  local seen = {}
  
  for _, auction in ipairs(myRows) do
    local itemKey = string.format("%d:%d", auction.itemId or 0, auction.stackSize or 1)
    
    if not seen[itemKey] then
      seen[itemKey] = true
      
      local marketData = GetInstantMarketData(auction.itemId, auction.stackSize, auction.link)
      
      if marketData then
        TrackerAuction.PersistMicroSnapshot(
          auction.itemId,
          auction.stackSize,
          marketData,
          nil
        )
        count = count + 1
      end
    end
  end
  
  return count
end

-- ============================================================================
-- ORIGINAL FUNCTIONALITY BELOW (PRESERVED 100%)
-- ============================================================================

-- ===== Lifecycle chat (colored) =====
local SCAN_LIFECYCLE = { active = false }
local function TA_Chat(msg)
  if DEFAULT_CHAT_FRAME and DEFAULT_CHAT_FRAME.AddMessage then
    DEFAULT_CHAT_FRAME:AddMessage(msg)
  else
    print(msg)
  end
end
local function TA_AnnounceScanStartedOnce()
  if SCAN_LIFECYCLE.active then return end
  SCAN_LIFECYCLE.active = true
  
  local msg = "|cff0070dd[WhoDAT]|r|cffffffff-|r|cff1eff00Auction Scan Started.|r - "
    .. "|cffff69b4Y|r|cffff0000o|r|cffff69b4u|r|cffff0000r|r "
    .. "|cffff69b4p|r|cffff0000a|r|cffff69b4t|r|cffff0000i|r|cffff69b4e|r|cffff0000n|r|cffff69b4c|r|cffff0000e|r "
    .. "|cffff69b4i|r|cffff0000s|r "
    .. "|cffff69b4a|r|cffff0000p|r|cffff69b4p|r|cffff0000r|r|cffff69b4e|r|cffff0000c|r|cffff69b4i|r|cffff0000a|r|cffff69b4t|r|cffff0000e|r|cffff69b4d|r|cffff0000!   <3|r"

  TA_Chat(msg)
end
local function TA_AnnounceScanCompletedOnce()
  if not SCAN_LIFECYCLE.active then return end
  SCAN_LIFECYCLE.active = false
  local msg = "|cff0070dd[WhoDAT]|r|cffffffff-|r|cff1eff00Auction Scan Completed!|r"
  TA_Chat(msg)
end

-- ===== Readiness (Blizzard AH API) =====
local function TA_EnsureAuctionReady()
  -- Ensure Blizzard Auction UI is loaded (BrowseButton..Seller fontstrings live here)
  if type(LoadAddOn) == "function" then pcall(function() LoadAddOn("Blizzard_AuctionUI") end) end
  local canQuery = false
  local ok, cq = pcall(function() return CanSendAuctionQuery() end)
  if ok then canQuery = cq end
  if canQuery then return true end
  if AuctionFrame and AuctionFrame:IsShown() then return true end
  return false
end

-- ===== Seller column reader (no tooltip, AH stays visible) =====
local function TA_ReadSellerFromBrowseRow(index)
  -- Blizzard creates BrowseButtonN Seller fontstrings when the browse list is rendered.
  -- Works on WotLK 3.3.5a (including Warmane/Icecrown).
  -- NOTE: Only the visible batch is available (<= 50); align with numBatch for current page.
  local fs = _G["BrowseButton"..index.."Seller"]
  if fs and fs.GetText then
    local s = fs:GetText()
    if s and s ~= "" then return s end
  end
  return nil
end

-- ===== Lightweight timer =====
local function TA_After(delay, fn)
  local f = CreateFrame("Frame")
  local acc = 0
  f:SetScript("OnUpdate", function(_, elapsed)
    acc = acc + elapsed
    if acc >= (delay or 0) then
      f:SetScript("OnUpdate", nil)
      pcall(fn)
    end
  end)
end

-- ===== Wait until CanSendAuctionQuery() is true (with deadline), then invoke =====
local function TA_WaitForCanQuery(maxWaitSec, onReady)
  local deadline = time() + (maxWaitSec or 5)
  local function poll()
    local ok, can = pcall(function() return CanSendAuctionQuery() end)
    if ok and can then pcall(onReady); return end
    if time() >= deadline then
      -- still proceed; some realms return false too aggressively
      pcall(onReady); return
    end
    TA_After(0.25, poll)
  end
  poll()
end

-- === Listing identity & upsert helpers ===
local function TA_RowKeyPartsFromLink(link)
  local itemString = link and link:match("item:[^\n]+")
  local fields = {}
  if itemString then
    for v in itemString:gmatch("([^:]+)") do table.insert(fields, v) end
  end
  local itemId   = tonumber(fields[2]) or 0
  local enchant  = tonumber(fields[3]) or 0
  local suffix   = tonumber(fields[7]) or 0
  local uniqueId = tonumber(fields[8]) or 0
  return itemId, enchant, suffix, uniqueId
end

-- NEW: normalize seller (case/whitespace)
local function TA_NormalizeSeller(s)
  if not s or s == "" then return "" end
  s = s:gsub("%s+", " "):gsub("%s+$", ""):gsub("^%s+", ""):lower()
  return s
end

-- NEW: canonical listing key (WITH uniqueId for individual auction tracking; price-aware; seller normalized)
local function TA_MakeListingKey(rowLike)
  local itemId, _, suffix, uniqueId = TA_RowKeyPartsFromLink(rowLike.link)
  local stackSize = rowLike.stackSize or 1
  local sellerNorm = TA_NormalizeSeller(rowLike.seller or rowLike.sellerName)
  local priceStack = rowLike.price
    or ((rowLike.buyoutPrice and rowLike.buyoutPrice > 0) and rowLike.buyoutPrice)
    or (rowLike.nextBid or 0) or 0
  -- Include uniqueId to track each individual auction listing separately (not just unique item/price combos)
  -- For owner auctions, uniqueId is 0, so use ownerSeq instead
  local finalId = (uniqueId and uniqueId > 0) and uniqueId or (rowLike.ownerSeq or 0)
  return string.format("%d:%d:%d:%s:%d:%d", itemId, suffix, stackSize, sellerNorm, priceStack, finalId)
end

local function TA_UpsertRow(bucket, rowLike)
  local newKey = TA_MakeListingKey(rowLike)
  for i = #bucket, 1, -1 do
    local oldKey = TA_MakeListingKey(bucket[i])
    if oldKey == newKey then
      local b = bucket[i]
      b.ts = time()
      b.duration = rowLike.duration or rowLike.timeLeft or b.duration
      b.price = rowLike.price
        or ((rowLike.buyoutPrice and rowLike.buyoutPrice > 0) and rowLike.buyoutPrice)
        or (rowLike.nextBid or b.price)
      b.link = rowLike.link or b.link
      b.name = rowLike.name or b.name
      b.stackSize = rowLike.stackSize or b.stackSize
      b.seller = (rowLike.seller or rowLike.sellerName or b.seller)
      return
    end
  end
  table.insert(bucket, {
    ts = time(),
    itemId = rowLike.itemId,
    ownerSeq = rowLike.ownerSeq,  -- Track individual auction sequence
    link = rowLike.link,
    name = rowLike.name,
    stackSize = rowLike.stackSize or 1,
    price = rowLike.price
      or ((rowLike.buyoutPrice and rowLike.buyoutPrice > 0) and rowLike.buyoutPrice)
      or (rowLike.nextBid or 0),
    duration = rowLike.duration or rowLike.timeLeft,
    seller = rowLike.seller or rowLike.sellerName,
    sold = false,
  })
end

-- NEW: Better upsert for owner auctions in persistent storage
-- Uses a more stable key that doesn't rely on ownerSeq
local function TA_UpsertOwnerAuction(bucket, rowLike)
  -- Create a stable key for owner auctions: itemId:stackSize:price:duration
  -- This identifies unique auction listings without using ownerSeq
  local stableKey = string.format("%d:%d:%d:%d",
    rowLike.itemId or 0,
    rowLike.stackSize or 1,
    rowLike.price or rowLike.buyoutPrice or 0,
    rowLike.duration or rowLike.timeLeft or 0
  )
  
  -- Look for existing auction with same stable key
  for i = #bucket, 1, -1 do
    local existingKey = string.format("%d:%d:%d:%d",
      bucket[i].itemId or 0,
      bucket[i].stackSize or 1,
      bucket[i].price or 0,
      bucket[i].duration or 0
    )
    
    if existingKey == stableKey and not bucket[i].sold and not bucket[i].expired then
      -- Update existing auction
      bucket[i].ts = time()
      bucket[i].link = rowLike.link or bucket[i].link
      bucket[i].name = rowLike.name or bucket[i].name
      bucket[i].seller = rowLike.seller or rowLike.sellerName or bucket[i].seller
      return false -- Not a new auction
    end
  end
  
  -- New auction - add it
  table.insert(bucket, {
    ts = time(),
    itemId = rowLike.itemId,
    ownerSeq = rowLike.ownerSeq,
    link = rowLike.link,
    name = rowLike.name,
    stackSize = rowLike.stackSize or 1,
    price = rowLike.price
      or ((rowLike.buyoutPrice and rowLike.buyoutPrice > 0) and rowLike.buyoutPrice)
      or (rowLike.nextBid or 0),
    duration = rowLike.duration or rowLike.timeLeft,
    seller = rowLike.seller or rowLike.sellerName,
    sold = false,
  })
  return true -- New auction added
end

-- NEW: session-level dedupe guard
local SESSION_SEEN = {}
local SESSION_TTL  = 15 -- seconds

local function TA_SessionMark(key)
  SESSION_SEEN[key] = time()
end
local function TA_SessionRecentlySeen(key)
  local t = SESSION_SEEN[key]
  return t and (time() - t) < SESSION_TTL
end
local function TA_SessionGC()
  local now = time()
  for k, ts in pairs(SESSION_SEEN) do
    if (now - ts) > (SESSION_TTL * 4) then SESSION_SEEN[k] = nil end
  end
end
-- PUBLIC: Expose GC for external callers (memory management)
function TrackerAuction.SessionGC()
  TA_SessionGC()
end
-- NEW: wrapped upsert that honors session dedupe
local function TA_UpsertRow_Dedup(bucket, rowLike)
  local key = TA_MakeListingKey(rowLike)
  if TA_SessionRecentlySeen(key) then
    -- Refresh ts if the row already exists; otherwise skip append
    for i = #bucket, 1, -1 do
      if TA_MakeListingKey(bucket[i]) == key then
        bucket[i].ts = time()
        return
      end
    end
    return
  end
  TA_UpsertRow(bucket, rowLike)
  TA_SessionMark(key)
end

-- ===== Duration buckets =====
local TIME_BUCKET_SECONDS = {
  [1] = 1800,  -- ~30 minutes
  [2] = 7200,  -- ~2 hours
  [3] = 43200, -- ~12 hours
  [4] = 172800,-- ~48 hours
}
function TrackerAuction.TimeBucketToSeconds(bucket)
  return TIME_BUCKET_SECONDS[bucket or 1] or 0
end
function TrackerAuction.TimeBucketToHours(bucket)
  local s = TrackerAuction.TimeBucketToSeconds(bucket)
  if s >= 172800 then return 48
  elseif s >= 43200 then return 12
  elseif s >= 7200 then return 2
  else return 0.5 end
end

-- Owner auction scan counter for uniqueness (since uniqueId is 0 for owner auctions)
local OWNER_SCAN_SEQUENCE = 0
local function TA_ResetOwnerScanSequence()
  OWNER_SCAN_SEQUENCE = 0
end
local function TA_NextOwnerSequence()
  OWNER_SCAN_SEQUENCE = OWNER_SCAN_SEQUENCE + 1
  return OWNER_SCAN_SEQUENCE
end

-- ===== Row normalizer (with seller backfill from Browse UI) =====
function TrackerAuction.NormalizeAuctionRow(list, index)
  local link = GetAuctionItemLink(list, index)
  if not link then return nil end
  local name, texture, count, quality, canUse, level,
        minBid, minIncrement, buyoutPrice, bidAmount,
        isHighBidder, owner, saleStatus = GetAuctionItemInfo(list, index)
  local timeLeft = GetAuctionItemTimeLeft(list, index)

  count = (count and count > 0) and count or 1
  minBid, minIncrement, buyoutPrice = minBid or 0, minIncrement or 0, buyoutPrice or 0
  bidAmount, owner = bidAmount or 0, owner or ""
  isHighBidder = not not isHighBidder

  local nextBid
  if bidAmount > 0 then
    nextBid = bidAmount + minIncrement
    if buyoutPrice > 0 and nextBid > buyoutPrice then nextBid = buyoutPrice end
  elseif minBid > 0 then
    nextBid = minBid
  else
    nextBid = 1
  end

  local _, _, _, itemLevel, itemType, itemSubType, _, itemEquipLoc = GetItemInfo(link)

  local timeSeen = time()

  local itemString = link:match("item:[^\n]+")
  local itemId, suffix, uniqueId, enchant = 0, 0, 0, 0
  if itemString then
    local fields = {}
    for v in itemString:gmatch("([^:]+)") do table.insert(fields, v) end
    itemId   = tonumber(fields[2]) or 0
    enchant  = tonumber(fields[3]) or 0
    suffix   = tonumber(fields[7]) or 0
    uniqueId = tonumber(fields[8]) or 0
  end

  -- Resilient seller name with tooltip fallback:
  local sellerName = owner
  if (not sellerName or sellerName == "") and list == "list" then
    -- Try 1: Read seller from the visible browse row seller column
    local s = TA_ReadSellerFromBrowseRow(index)
    if s and s ~= "" then 
      sellerName = s
    else
      -- Try 2: Extract from tooltip as fallback
      if GameTooltip then
        GameTooltip:SetOwner(UIParent, "ANCHOR_NONE")
        GameTooltip:SetAuctionItem(list, index)
        -- Scan tooltip lines for seller (format varies by locale)
        for i = 1, GameTooltip:NumLines() do
          local line = _G["GameTooltipTextLeft"..i]
          if line then
            local text = line:GetText()
            if text then
              -- Try common patterns: "Seller: Name" or just the name alone
              local seller = text:match("Seller:%s*(.+)") or text:match("Owner:%s*(.+)")
              if seller and seller ~= "" then
                sellerName = seller
                break
              end
            end
          end
        end
        GameTooltip:Hide()
      end
      
      -- NEW: Try 3: Auctioneer as final fallback for seller name
      if (not sellerName or sellerName == "") then
        local auctSeller = GetAuctioneerSeller(link)
        if auctSeller and auctSeller ~= "" then
          sellerName = auctSeller
        end
      end
    end
  end
  
  -- For owner auctions, assign unique sequence number (uniqueId is always 0)
  local ownerSeq = (list == "owner") and TA_NextOwnerSequence() or 0

  -- NEW: Validate price if we have Auctioneer (sanity check, doesn't change output)
  ValidatePrice(link, buyoutPrice)

  -- IMPORTANT: Return structure is UNCHANGED from original
  -- No new fields added - Auctioneer only used to improve existing field quality
  return {
    link = link, name = name, itemId = itemId, suffix = suffix, uniqueId = uniqueId, enchant = enchant, ownerSeq = ownerSeq,
    texture = texture, quality = quality, itemType = itemType, subType = itemSubType, equipLoc = itemEquipLoc,
    stackSize = count, minBid = minBid, increment = minIncrement, buyoutPrice = buyoutPrice,
    curBid = bidAmount, nextBid = nextBid, isHighBidder = isHighBidder, sellerName = sellerName,
    timeLeft = timeLeft, timeSeen = timeSeen,
    -- Fields for sold/expired tracking (ORIGINAL)
    ts = timeSeen,           -- Timestamp (alias of timeSeen for consistency)
    duration = timeLeft,     -- Duration code (1-4) for expiry calculation
    seller = sellerName,     -- Seller name (alias for consistency)
    price = buyoutPrice,     -- Price (alias for consistency)
    sold = false,            -- Not sold yet
    sold_ts = nil,           -- Will be set when sold
    sold_price = nil,        -- Will be set from mail
    expired = false,         -- Not expired yet
    expired_ts = nil,        -- Will be set when expired
  }
end

-- ===== Client-side filter =====
function TrackerAuction._passes(rec, params)
  if not params then return true end
  if params.sellerIsMe and rec.sellerName ~= UnitName("player") then return false end
  if params.name then
    local needle = TrackerAuction.QuerySafeName(params.name)
    if needle then
      local rn = rec.name and rec.name:lower() or ""
      if not rn:find(needle, 1, true) then return false end
    end
  end
  if params.itemIdExact and rec.itemId ~= params.itemIdExact then return false end
  if params.minStack and rec.stackSize < params.minStack then return false end
  if params.maxStack and rec.stackSize > params.maxStack then return false end
  local price = (rec.buyoutPrice > 0) and rec.buyoutPrice or rec.nextBid
  if params.perItem then
    local ppi = (price > 0) and math.floor(price / rec.stackSize + 0.5) or 0
    if params.minPrice and ppi < params.minPrice then return false end
    if params.maxPrice and ppi > params.maxPrice then return false end
  else
    if params.minPrice and price < params.minPrice then return false end
    if params.maxPrice and price > params.maxPrice then return false end
  end
  return true
end

-- ===== Visible page quick read (with seller backfill per index) =====
function TrackerAuction.QueryVisiblePage(params)
  local list = "list"
  local numBatch, total = GetNumAuctionItems(list)
  local rows = {}
  for i = 1, numBatch do
    local rec = TrackerAuction.NormalizeAuctionRow(list, i)
    if rec and TrackerAuction._passes(rec, params) then
      -- If seller still blank, attempt UI seller column read
      if (not rec.sellerName or rec.sellerName == "") then
        local s = TA_ReadSellerFromBrowseRow(i)
        if s then rec.sellerName = s; rec.seller = s end
      end
      table.insert(rows, rec)
    end
  end
  local stats = { source="visiblePage", totalFound=#rows, numBatch=numBatch, totalAuctions=total, when=time() }
  return rows, stats
end

-- ===== Owner fast path =====
local function TA_ReadOwnerRows()
  local list = "owner"
  TA_ResetOwnerScanSequence() -- Reset counter for this scan
  local numOwner = GetNumAuctionItems(list)
  local rows = {}
  for i = 1, numOwner do
    local rec = TrackerAuction.NormalizeAuctionRow(list, i)
    if rec then 
      table.insert(rows, rec)
    end
  end
  return rows, numOwner
end

-- ===== Market helpers =====
function TrackerAuction.MakeItemKey(rec)
  return string.format("%d:%d", rec.itemId or 0, rec.stackSize or 1)
end

-- ===== Compute low/high price bands from market rows =====
function TrackerAuction.ComputeLowHigh(rows, take)
  take = take or 3
  local sorted = {}
  
  -- Copy and sort by price
  for i = 1, #rows do
    if rows[i] and rows[i].price then
      table.insert(sorted, rows[i])
    end
  end
  
  table.sort(sorted, function(a, b)
    return (a.price or 0) < (b.price or 0)
  end)
  
  local low, high = {}, {}
  
  -- Helper function to extract seller with fallbacks
local function ExtractSeller(row)
  -- Try direct seller field
  if row.seller and row.seller ~= "" then
    return row.seller
  end
  
  -- Try sellerName field
  if row.sellerName and row.sellerName ~= "" then
    return row.sellerName
  end
  
  -- No more attempts - just return Unknown
  return "Unknown"
end
  
  -- Take bottom N (lowest prices)
  for i = 1, math.min(take, #sorted) do
    local r = sorted[i]
    table.insert(low, {
      priceStack = r.price,
      priceItem = math.floor(r.price / (r.stackSize > 0 and r.stackSize or 1)),
      seller = ExtractSeller(r),
      link = r.link or ""
    })
  end
  
  -- Take top N (highest prices)
  local start = math.max(1, #sorted - take + 1)
  for i = #sorted, start, -1 do
    local r = sorted[i]
    table.insert(high, {
      priceStack = r.price,
      priceItem = math.floor(r.price / (r.stackSize > 0 and r.stackSize or 1)),
      seller = ExtractSeller(r),
      link = r.link or ""
    })
  end
  
  return { low = low, high = high }
end

-- ===== Market query (async) - Wrath signature + gating + timeout fallback =====
do
  local QUERY = { active=false, params=nil, rows={}, page=0, total=0, onComplete=nil, waiting=false }
  local PER_PAGE = 50
  local FALLBACK_AFTER = 2.0 -- seconds without AUCTION_ITEM_LIST_UPDATE

  -- Wrath 3.3.5: QueryAuctionItems(name, minLevel, maxLevel, page, isUsable, qualityIndex, getAll, exactMatch, filterData)
  local function callQueryAuctionItems(name, page, exactMatch, getAll)
    local minLevel, maxLevel = 0, 0
    local isUsable, qualityIndex = false, nil
    local filterData = nil
    pcall(function()
      QueryAuctionItems(name, minLevel, maxLevel, page or 0, isUsable, qualityIndex, getAll or false, exactMatch or false, filterData)
    end)
  end

  local function issueQuery(page)
    if not TA_EnsureAuctionReady() then return end
    local p = QUERY.params or {}
    local nameExact = p.name
    if ((not nameExact or nameExact == "") and p.itemIdExact) then
      local itemName = GetItemInfo(p.itemIdExact)
      if itemName then nameExact = itemName end
    end
    local exactMatch = nameExact and nameExact ~= ""
    callQueryAuctionItems(exactMatch and nameExact or nil, page or 0, exactMatch, false)
    -- start timeout
    QUERY.waiting = true
    local expectedPage = page or 0
    TA_After(FALLBACK_AFTER, function()
      if QUERY.active and QUERY.waiting and QUERY.page == expectedPage then
        local function fallbackVisiblePageRead()
          local rows,_ = TrackerAuction.QueryVisiblePage({
            name = nameExact,
            itemIdExact = p.itemIdExact,
            minStack = p.minStack,
            maxStack = p.maxStack,
          })
          rows = rows or {}
          local added = 0
          for i = 1, #rows do
            local rec = rows[i]
            if rec and rec.sellerName ~= UnitName("player") then
              table.insert(QUERY.rows, rec); added = added + 1
            end
          end
          if added == 0 then
            TA_After(0.5, function()
              local rows2,_ = TrackerAuction.QueryVisiblePage({
                name = nameExact,
                itemIdExact = p.itemIdExact,
                minStack = p.minStack,
                maxStack = p.maxStack,
              })
              rows2 = rows2 or {}
              for i = 1, #rows2 do
                local rec = rows2[i]
                if rec and rec.sellerName ~= UnitName("player") then
                  table.insert(QUERY.rows, rec)
                end
              end
              TrackerAuction._ContinueAfterPageTimeout()
            end)
          else
            TrackerAuction._ContinueAfterPageTimeout()
          end
        end
        fallbackVisiblePageRead()
      end
    end)
  end

  function TrackerAuction._ContinueAfterPageTimeout()
    local takeNeeded = (QUERY.params and QUERY.params._take) or 3
    local haveEnough = #QUERY.rows >= (2 * takeNeeded)
    if haveEnough then
      local cb = QUERY.onComplete
      local outRows = QUERY.rows
      QUERY.active, QUERY.params, QUERY.rows, QUERY.page, QUERY.total, QUERY.onComplete, QUERY.waiting =
        false, nil, {}, 0, 0, nil, false
      if type(cb) == "function" then cb(outRows) end
    else
      QUERY.page = QUERY.page + 1
      TA_WaitForCanQuery(5, function() TA_After(0.1, function() issueQuery(QUERY.page) end) end)
    end
  end

  local f = CreateFrame("Frame")
  f:RegisterEvent("AUCTION_ITEM_LIST_UPDATE")
  f:SetScript("OnEvent", function(_, event)
    if not QUERY.active then return end
    local list = "list"
    local numBatch, total = GetNumAuctionItems(list)
    QUERY.total = total or 0
    QUERY.waiting = false
    for i = 1, (numBatch or 0) do
      local rec = TrackerAuction.NormalizeAuctionRow(list, i)
      if rec and TrackerAuction._passes(rec, QUERY.params) then
        -- If seller blank, backfill from Browse UI while AH is visible
        if (not rec.sellerName or rec.sellerName == "") then
          local s = TA_ReadSellerFromBrowseRow(i)
          if s then rec.sellerName = s; rec.seller = s end
        end
        if rec.sellerName ~= UnitName("player") then
          table.insert(QUERY.rows, rec)
        end
      end
    end
    local takeNeeded = (QUERY.params and QUERY.params._take) or 3
    local haveEnough = #QUERY.rows >= (2 * takeNeeded)
    local pagesTotal = math.floor(math.max(0, QUERY.total - 1) / PER_PAGE)
    if haveEnough or QUERY.page >= pagesTotal then
      local cb = QUERY.onComplete
      local rows = QUERY.rows
      QUERY.active, QUERY.params, QUERY.rows, QUERY.page, QUERY.total, QUERY.onComplete =
        false, nil, {}, 0, 0, nil
      if type(cb) == "function" then cb(rows) end
      return
    end
    QUERY.page = QUERY.page + 1
    TA_WaitForCanQuery(5, function() TA_After(0.1, function() issueQuery(QUERY.page) end) end)
  end)

  -- NEW: Piggyback on Auctioneer scan (listen-only mode)
  function TrackerAuction.ScanMarketAsync(params, onComplete)
    if not TA_EnsureAuctionReady() then
      if type(onComplete) == "function" then onComplete({}, { error = "AuctionHouseNotReady" }) end
      return nil, "AuctionHouseNotReady"
    end

-- PIGGYBACK PATH: Let Auctioneer scan, we just listen
if activeAuctionSource == "Auctioneer" and CONFIG.PIGGYBACK_AUCTIONEER_SCAN then
  local pRoot = params or {}
  
  -- Enforce same stack size comparisons
  if pRoot.minStack == nil and pRoot.maxStack == nil and pRoot.stackSize then
    pRoot.minStack = pRoot.stackSize
    pRoot.maxStack = pRoot.stackSize
  end
  
  local ACfgRoot = _G.WhoDAT_GetAuctionCfg and _G.WhoDAT_GetAuctionCfg() or (_G.WhoDAT_Config or {})
  local ACfg = ACfgRoot.auction or ACfgRoot
  local marketTake = (type(pRoot.marketTake) == "number" and pRoot.marketTake > 0) and pRoot.marketTake or (ACfg.market_take or 3)
  
  -- NEW: Trigger Auctioneer to scan
  local scanStarted = TriggerAuctioneerScan()
  if not scanStarted then
    print("[WhoDAT] Warning: Could not trigger Auctioneer scan automatically")
    print("[WhoDAT] Please start an Auctioneer scan manually, then WhoDAT will piggyback")
  end
  
  -- Session state for piggyback listening
  local rows = {}
  local lastTick = time()
  local IDLE_CUTOFF = 1.5  -- seconds since last page load
  local frame = CreateFrame("Frame")
  
  local function finish()
    frame:SetScript("OnUpdate", nil)
    frame:SetScript("OnEvent", nil)
    frame:UnregisterAllEvents()
    
    if type(onComplete) == "function" then 
      onComplete(rows, { piggyback=true, source="Auctioneer", count=#rows }) 
    end
  end
  
  frame:RegisterEvent("AUCTION_ITEM_LIST_UPDATE")
  frame:SetScript("OnEvent", function(_, event)
    if event ~= "AUCTION_ITEM_LIST_UPDATE" then return end
    lastTick = time()
    
    local list = "list"
    local numBatch, _ = GetNumAuctionItems(list)
    
    for i = 1, (numBatch or 0) do
      local rec = TrackerAuction.NormalizeAuctionRow(list, i)
      if rec and TrackerAuction._passes(rec, pRoot) then
        -- Backfill seller if visible column has it
        if (not rec.sellerName or rec.sellerName == "") then
          local s = TA_ReadSellerFromBrowseRow(i)
          if s then rec.sellerName = s; rec.seller = s end
        end
        
        -- Ignore my own auctions in market rows
        if rec.sellerName ~= UnitName("player") then
          table.insert(rows, rec)
        end
      end
    end
  end)
  
  -- Idle detector: if Auctioneer stops loading pages, we finish
  frame:SetScript("OnUpdate", function(_, elapsed)
    local scanning = IsAuctioneerBusy()
    if (not scanning) and (time() - lastTick) > IDLE_CUTOFF then
      finish()
    end
  end)
  
  print("[WhoDAT] Piggybacking on Auctioneer scan - waiting for data...")
  return true
end

    -- ORIGINAL PATH: Manual QueryAuctionItems (when no Auctioneer)
    local pRoot = params or {}
    if pRoot.minStack == nil and pRoot.maxStack == nil and pRoot.stackSize then
      pRoot.minStack = pRoot.stackSize
      pRoot.maxStack = pRoot.stackSize
    end
    local ACfgRoot = _G.WhoDAT_GetAuctionCfg and _G.WhoDAT_GetAuctionCfg() or (_G.WhoDAT_Config or {})
    local ACfg = ACfgRoot.auction or ACfgRoot
    local marketTake = (type(pRoot.marketTake) == "number" and pRoot.marketTake > 0) and pRoot.marketTake or (ACfg.market_take or 3)
    pRoot._take = marketTake

    QUERY.active = true
    QUERY.params = pRoot
    QUERY.rows = {}
    QUERY.page = 0
    QUERY.total = 0
    QUERY.onComplete = onComplete
    TA_WaitForCanQuery(5, function() issueQuery(0) end)
    return true
  end
end
-- ===== Browse UI Market Scanner (gets seller names without Auctioneer) =====
function TrackerAuction.ScanMarketViaBrowseUI(options, onComplete)
  options = options or {}
  local itemIdExact = options.itemIdExact
  local stackSize = options.stackSize
  local marketTake = options.marketTake or 3
  
  if not itemIdExact then
    if type(onComplete) == "function" then
      onComplete({}, { error = "NoItemId" })
    end
    return
  end
  
  -- Get item name for searching
  local itemName = GetItemInfo(itemIdExact)
  if not itemName then
    -- Item not cached yet, try again
    TA_After(0.1, function()
      TrackerAuction.ScanMarketViaBrowseUI(options, onComplete)
    end)
    return
  end
  
  -- Make sure we're on Browse tab
  if AuctionFrame.selectedTab ~= 1 then
    AuctionFrameTab1:Click()
    TA_After(0.2, function()
      TrackerAuction.ScanMarketViaBrowseUI(options, onComplete)
    end)
    return
  end
  
  local allRows = {}
  local currentPage = 0
  local maxPages = 3  -- Don't scan forever
  
  local function scanNextPage()
    local numBatch, totalAuctions = GetNumAuctionItems("list")
    
    if not numBatch or numBatch == 0 then
      -- Done - return results
      if type(onComplete) == "function" then
        onComplete(allRows, { 
          pages = currentPage,
          total = #allRows,
          method = "browse_ui"
        })
      end
      return
    end
    
    -- Read all visible rows on this page
    for i = 1, numBatch do
      local rec = TrackerAuction.NormalizeAuctionRow("list", i)
      
      -- Filter by itemId and stackSize
      if rec and rec.itemId == itemIdExact then
        if not stackSize or rec.stackSize == stackSize then
          table.insert(allRows, rec)
        end
      end
    end
    
    currentPage = currentPage + 1
    
    -- Check if we have enough or hit max pages
    if #allRows >= 50 or currentPage >= maxPages then
      if type(onComplete) == "function" then
        onComplete(allRows, { 
          pages = currentPage,
          total = #allRows,
          method = "browse_ui"
        })
      end
      return
    end
    
    -- Try to get next page
    local hasMore = numBatch == NUM_AUCTION_ITEMS_PER_PAGE
    if hasMore then
      local canQuery = CanSendAuctionQuery("list")
      if canQuery then
        QueryAuctionItems(itemName, nil, nil, 0, 0, 0, currentPage, false, false)
        TA_After(0.7, scanNextPage)
      else
        -- Query on cooldown, return what we have
        if type(onComplete) == "function" then
          onComplete(allRows, { 
            pages = currentPage,
            total = #allRows,
            method = "browse_ui",
            incomplete = true
          })
        end
      end
    else
      -- No more pages
      if type(onComplete) == "function" then
        onComplete(allRows, { 
          pages = currentPage,
          total = #allRows,
          method = "browse_ui"
        })
      end
    end
  end
  
  -- Start initial search by item name
  local canQuery = CanSendAuctionQuery("list")
  if not canQuery then
    if type(onComplete) == "function" then
      onComplete({}, { error = "QueryOnCooldown" })
    end
    return
  end
  
  -- Search for the item by name
  BrowseName:SetText(itemName)
  QueryAuctionItems(itemName, nil, nil, 0, 0, 0, 0, false, false)
  
  -- Wait for results
  TA_After(0.7, scanNextPage)
end

-- ===== Market scan =====
function TrackerAuction.ScanMyAuctionsWithMarket(options, onComplete)
  options = options or {}
  local ACfgRoot = _G.WhoDAT_GetAuctionCfg and _G.WhoDAT_GetAuctionCfg() or (_G.WhoDAT_Config or {})
  local ACfg = ACfgRoot.auction or ACfgRoot
  local persistTS = (options.persistTS ~= nil) and options.persistTS or (ACfg.persist_market_ts ~= false)
  local marketTake = (type(options.marketTake) == "number" and options.marketTake > 0) and options.marketTake or (ACfg.market_take or 3)
  local persistRows = (ACfg.persist_rows_to_sv ~= false)

  if not TA_EnsureAuctionReady() then
    if type(onComplete) == "function" then onComplete({}, { error = "AuctionHouseNotReady" }) end
    return nil, "AuctionHouseNotReady"
  end

  -- Announce start once per scan lifecycle
  TA_AnnounceScanStartedOnce()

  -- 1) Owner list first
  local myRows, numOwner = TA_ReadOwnerRows()

  -- 2) Worklist (distinct itemId + stackSize)
  local worklist, seen = {}, {}
  for i = 1, #myRows do
    local r = myRows[i]
    local stack = (r and r.stackSize) or 0
    if stack > 0 then
      local key = TrackerAuction.MakeItemKey(r)
      if key and not seen[key] then
        seen[key] = true
        table.insert(worklist, { 
          itemId = r.itemId, 
          stackSize = r.stackSize, 
          link = r.link,
          myPrice = r.price  -- NEW: Store user's price
        })
      end
    end
  end

  -- 3) Persist owner list to global DB (OUTPUT UNCHANGED)
  local realm  = GetRealmName() or "UNKNOWN"
  local faction= UnitFactionGroup("player") or "Neutral"
  local char   = UnitName("player") or "UNKNOWN"
  local key    = realm .. "-" .. faction .. ":" .. char

  WhoDAT_AuctionDB = WhoDAT_AuctionDB or {}
  WhoDAT_AuctionDB[key] = WhoDAT_AuctionDB[key] or {}
  local bucket = WhoDAT_AuctionDB[key]

  for i = 1, #myRows do
    TA_UpsertOwnerAuction(bucket, myRows[i])
  end

  if #worklist == 0 then
    TA_AnnounceScanCompletedOnce()
    if type(onComplete) == "function" then onComplete({}, { message = "NoOwnedAuctions" }) end
    return
  end

  -- 4) Market scan strategy depends on piggyback mode
  
  -- NEW: PIGGYBACK MODE - Single scan for all items
  if activeAuctionSource == "Auctioneer" and CONFIG.PIGGYBACK_AUCTIONEER_SCAN then
    print("[WhoDAT] Piggybacking on Auctioneer - scanning all items at once...")
    
    -- Do ONE scan without item filters (Auctioneer scans everything)
    TrackerAuction.ScanMarketAsync({
      -- No item filter - get everything
      marketTake = marketTake,
    }, function(allRows, stats)
      -- Now process results for each item in worklist
      local allMarket = {}
      
      for _, item in ipairs(worklist) do
        -- Filter rows for this specific item
        local itemRows = {}
        for _, row in ipairs(allRows) do
          if row.itemId == item.itemId and row.stackSize == item.stackSize then
            table.insert(itemRows, row)
          end
        end
        
        -- Compute market data for this item
        local mkt = TrackerAuction.ComputeLowHigh(itemRows, marketTake)
        table.insert(allMarket, { 
          itemId = item.itemId, 
          stackSize = item.stackSize, 
          link = item.link, 
          market = mkt 
        })
        
        -- Persist time-series with user's price
        if persistTS then
          TrackerAuction.PersistMicroSnapshot(
            item.itemId, 
            item.stackSize, 
            mkt, 
            persistRows and itemRows or nil,
            item.myPrice  -- NEW: Pass user's auction price
          )
        end
      end
      
      TA_SessionGC()
      TA_AnnounceScanCompletedOnce()
      if type(onComplete) == "function" then 
        onComplete(allMarket, { 
          numItems = #worklist, 
          totalRows = #allRows,
          piggyback = true 
        }) 
      end
    end)
    
    return
  end
  
-- ORIGINAL MODE - Per-item scanning (no Auctioneer or piggyback disabled)
local completed, total = 0, #worklist
local allMarket = {}

local function doNextItem()
  local idx = completed + 1
  if idx > total then
    TA_SessionGC()
    TA_AnnounceScanCompletedOnce()
    if type(onComplete) == "function" then onComplete(allMarket, { numItems = total }) end
    return
  end

  local item = worklist[idx]
  
  -- NEW: Always use Browse UI to get seller names
  TrackerAuction.ScanMarketViaBrowseUI({
    itemIdExact = item.itemId,
    stackSize   = item.stackSize,
    marketTake  = marketTake,
  }, function(mRows, stats)
    local mkt = TrackerAuction.ComputeLowHigh(mRows, marketTake)
    table.insert(allMarket, { itemId = item.itemId, stackSize = item.stackSize, link = item.link, market = mkt })

    -- Persist time-series with user's price
    if persistTS then
      TrackerAuction.PersistMicroSnapshot(
        item.itemId, 
        item.stackSize, 
        mkt, 
        persistRows and mRows or nil,
        item.myPrice
      )
    end

    completed = completed + 1
    
    -- Add delay between items to let UI settle
    TA_After(0.5, doNextItem)
  end)
end

doNextItem()
end

-- ===== Time-series micro-snapshots (FIXED: Flat key structure for PHP) =====
function TrackerAuction.PersistMicroSnapshot(itemId, stackSize, market, rows, myAuctionPrice)
  WhoDAT_AuctionMarketTS = WhoDAT_AuctionMarketTS or {}

  local realm   = GetRealmName() or "UNKNOWN"
  local faction = UnitFactionGroup("player") or "Neutral"
  
  -- Get item name and link
  local itemName = GetItemInfo(itemId) or "Unknown"
  local itemLink = market.low and market.low[1] and market.low[1].link or ""
  
  -- Create FLAT key as PHP expects: "Realm-Faction\nitemId:0:stackSize:itemName:price"
  local flatKey = string.format("%s-%s\n%d:0:%d:%s:%d",
    realm,
    faction,
    itemId,
    stackSize,
    itemName,
    myAuctionPrice or 0
  )
  
  WhoDAT_AuctionMarketTS[flatKey] = WhoDAT_AuctionMarketTS[flatKey] or {}

  local entry = {
    ts = time(),
    low = market.low or {},
    high = market.high or {},
  }
  
  -- Add "my" field if user has auction at this price
  if myAuctionPrice and myAuctionPrice > 0 then
    entry.my = {
      priceStack = myAuctionPrice,
      priceItem = math.floor(myAuctionPrice / (stackSize > 0 and stackSize or 1))
    }
  end
  
  -- if rows then
  --  entry.rows = rows
  -- end

  table.insert(WhoDAT_AuctionMarketTS[flatKey], entry)

  -- Compact if >200 entries
  if #WhoDAT_AuctionMarketTS[flatKey] > 200 then
    TrackerAuction.CompactTimeSeries(WhoDAT_AuctionMarketTS[flatKey])
  end
end

function TrackerAuction.CompactTimeSeries(bucket)
  -- Keep last 50; downsample older to hourly
  if #bucket <= 100 then return end
  table.sort(bucket, function(a,b) return (a.ts or 0) < (b.ts or 0) end)
  local keep = {}
  local now = time()
  local hourAgo = now - 3600
  for i = #bucket, 1, -1 do
    local e = bucket[i]
    if #keep < 50 then
      table.insert(keep, 1, e)
    elseif (e.ts or 0) >= hourAgo then
      table.insert(keep, 1, e)
    elseif (#keep % 4 == 0) then
      table.insert(keep, 1, e)
    end
  end
  for i = 1, #bucket do bucket[i] = nil end
  for i = 1, #keep do bucket[i] = keep[i] end
end

function TrackerAuction.QuerySafeName(raw)
  if not raw or raw == "" then return nil end
  local s = raw:lower():gsub("[^%w%s]", "")
  return s
end

-- ============================================================================
-- MAIL TRACKING SYSTEM (Original functionality preserved)
-- ============================================================================
TrackerAuction.MailTracker = TrackerAuction.MailTracker or {}
local MailTracker = TrackerAuction.MailTracker

-- NEW: Popup dialog for manual scan confirmation
StaticPopupDialogs["WHODAT_MARKET_SCAN_CONFIRM"] = {
  text = "WhoDAT: Scan Auction House for market prices?\n\nThis will query multiple pages and may take 30-60 seconds.",
  button1 = "Scan Now",
  button2 = "Cancel",
  OnAccept = function()
    print("[WhoDAT] Starting market scan...")
    TrackerAuction.ScanMyAuctionsWithMarket({}, function(results, stats)
      if stats and stats.error then
        print(string.format("[WhoDAT] Scan failed: %s", stats.error))
      elseif stats and stats.message then
        print(string.format("[WhoDAT] %s", stats.message))
      else
        print(string.format("[WhoDAT] Market scan completed! Scanned %d item(s)", stats and stats.numItems or 0))
      end
    end)
  end,
  timeout = 0,
  whileDead = true,
  hideOnEscape = true,
  preferredIndex = 3,
}

-- Parse auction mail to detect sales
local function ParseAuctionMail(mailIndex)
    local _, _, sender, subject, money, _, _, _, _, _, _, _ = GetInboxHeaderInfo(mailIndex)
    
    -- AH mail sender check (localized)
    if not sender then return nil end
    local isAHMail = sender:find("Auctioneer") or sender:find("Auction House")
    if not isAHMail then return nil end
    
    -- Must have gold
    if not money or money == 0 then return nil end
    
    -- Try to get item name from subject
    local itemName = subject and subject:match(":%s*(.+)") or subject
    
    -- Get actual item from mail attachment (more reliable)
    local itemLink = GetInboxItemLink(mailIndex, 1)
    if itemLink then
        itemName = itemLink:match("%[(.+)%]") or itemName
    end
    
    return {
        item_name = itemName,
        item_link = itemLink or itemName,
        gold_received = money,
        ts = time()
    }
end

-- Scan inbox for auction sales
function MailTracker:ScanInbox()
    local numMail = GetInboxNumItems()
    if numMail == 0 then return end
    
    local sales = {}
    
    for i = 1, numMail do
        local saleData = ParseAuctionMail(i)
        if saleData then
            table.insert(sales, saleData)
            self:MarkAuctionSold(saleData)
        end
    end
    
    return sales
end

-- Mark an auction as sold based on item match
function MailTracker:MarkAuctionSold(saleData)
    local realm  = GetRealmName() or "UNKNOWN"
    local faction= UnitFactionGroup("player") or "Neutral"
    local char   = UnitName("player") or "UNKNOWN"
    local key    = realm .. "-" .. faction .. ":" .. char
    
    if not WhoDAT_AuctionDB or not WhoDAT_AuctionDB[key] then
        return
    end
    
    local bucket = WhoDAT_AuctionDB[key]
    
    -- Find matching auction and mark it sold (OUTPUT UNCHANGED)
    for i, auction in ipairs(bucket) do
        if auction.name == saleData.item_name and not auction.sold then
            auction.sold = true
            auction.sold_ts = saleData.ts
            auction.sold_price = saleData.gold_received
            
            print(string.format("[WhoDAT] Auction SOLD: %s for %dg %ds %dc", 
                saleData.item_name,
                math.floor(saleData.gold_received / 10000),
                math.floor((saleData.gold_received % 10000) / 100),
                saleData.gold_received % 100
            ))
            
            break
        end
    end
end

-- ============================================================================
-- SNAPSHOT COMPARISON FOR SOLD/EXPIRED DETECTION (Original functionality)
-- ============================================================================
TrackerAuction.previousAuctions = TrackerAuction.previousAuctions or {}

function TrackerAuction.DetectAuctionChanges(currentAuctions)
    local sold = {}
    local expired = {}
    
    -- Create lookup table for current auctions using unique key
    local currentLookup = {}
    for _, auction in ipairs(currentAuctions) do
        local key = string.format("%d_%d_%d_%d", 
            auction.itemId or 0,
            auction.stackSize or 1,
            auction.buyoutPrice or 0,
            auction.duration or 0
        )
        currentLookup[key] = auction
    end
    
    -- Check previous auctions against current
    for _, prevAuction in ipairs(TrackerAuction.previousAuctions) do
        local key = string.format("%d_%d_%d_%d", 
            prevAuction.itemId or 0,
            prevAuction.stackSize or 1,
            prevAuction.buyoutPrice or 0,
            prevAuction.duration or 0
        )
        
        if not currentLookup[key] then
            -- Auction is gone - determine if sold or expired
            local timeLeft = prevAuction.duration or 4
            local hoursSince = (time() - (prevAuction.ts or time())) / 3600
            
            -- Duration: 1=short(2h), 2=medium(8h), 3=long(24h), 4=very long(48h)
            local maxHours = {[1]=2, [2]=8, [3]=24, [4]=48}
            local expectedExpiry = maxHours[timeLeft] or 48
            
            if hoursSince >= (expectedExpiry - 0.5) then
                table.insert(expired, prevAuction)
            else
                table.insert(sold, prevAuction)
            end
        end
    end
    
    -- Update previous snapshot
    TrackerAuction.previousAuctions = currentAuctions
    
    return sold, expired
end

function TrackerAuction.UpdateOwnerAuctionsWithChangeDetection()
    local currentAuctions, numOwner = TA_ReadOwnerRows()
    
    -- Detect what changed
    local sold, expired = TrackerAuction.DetectAuctionChanges(currentAuctions)
    
    -- Mark sold/expired auctions in database (OUTPUT UNCHANGED)
    local realm  = GetRealmName() or "UNKNOWN"
    local faction= UnitFactionGroup("player") or "Neutral"
    local char   = UnitName("player") or "UNKNOWN"
    local key    = realm .. "-" .. faction .. ":" .. char
    
    if WhoDAT_AuctionDB and WhoDAT_AuctionDB[key] then
        local bucket = WhoDAT_AuctionDB[key]
        
        -- Mark sold
        for _, soldAuction in ipairs(sold) do
            for i, auction in ipairs(bucket) do
                if auction.itemId == soldAuction.itemId 
                   and auction.stackSize == soldAuction.stackSize
                   and not auction.sold 
                   and not auction.expired then
                    auction.sold = true
                    auction.sold_ts = time()
                    print(string.format("[WhoDAT] Detected SOLD (snapshot): %s", auction.name or "Unknown"))
                    break
                end
            end
        end
        
        -- Mark expired
        for _, expiredAuction in ipairs(expired) do
            for i, auction in ipairs(bucket) do
                if auction.itemId == expiredAuction.itemId 
                   and auction.stackSize == expiredAuction.stackSize
                   and not auction.sold 
                   and not auction.expired then
                    auction.expired = true
                    auction.expired_ts = time()
                    print(string.format("[WhoDAT] Detected EXPIRED (snapshot): %s", auction.name or "Unknown"))
                    break
                end
            end
        end
    end
    
    return currentAuctions, numOwner
end

-- ============================================================================
-- EVENT REGISTRATION (Original functionality preserved)
-- ============================================================================

-- Mail tracking events
local mailFrame = CreateFrame("Frame")
mailFrame:RegisterEvent("MAIL_INBOX_UPDATE")
mailFrame:RegisterEvent("MAIL_CLOSED")

mailFrame:SetScript("OnEvent", function(self, event)
    if event == "MAIL_INBOX_UPDATE" then
        if TrackerAuction.MailTracker and TrackerAuction.MailTracker.ScanInbox then
            TrackerAuction.MailTracker:ScanInbox()
        end
    end
end)

-- NEW: Passive listener for Auctioneer scans
local passiveListener = nil

-- NEW: Passive listener for Auctioneer scans
local passiveListener = nil

local function SetupPassiveListener()
  if passiveListener then return end  -- Already set up
  
  passiveListener = CreateFrame("Frame")
  passiveListener:RegisterEvent("AUCTION_ITEM_LIST_UPDATE")
  
  local lastCapture = {}
  
  passiveListener:SetScript("OnEvent", function(_, event)
    -- Only capture when Auctioneer is scanning
    if not IsAuctioneerBusy() then return end
    
    local list = "list"
    local numBatch, _ = GetNumAuctionItems(list)
    
    if not numBatch or numBatch == 0 then return end
    
    -- Capture all visible rows
    local capturedByItem = {}
    
    for i = 1, numBatch do
      local rec = TrackerAuction.NormalizeAuctionRow(list, i)
      if rec and rec.sellerName ~= UnitName("player") then
        
        -- NEW OPTION A: Try Auctioneer seller data first
        if (not rec.sellerName or rec.sellerName == "") and _G.AucAdvanced then
          local ok, seller = pcall(function()
            -- Try to get seller from Auctioneer's scan data
            if AucAdvanced.Scan and AucAdvanced.Scan.GetImage then
              local image = AucAdvanced.Scan.GetImage()
              if image and image[i] then
                return image[i].owner
              end
            end
            return nil
          end)
          if ok and seller and seller ~= "" then
            rec.sellerName = seller
            rec.seller = seller
          end
        end
        
        -- Fallback to Browse UI seller extraction
        if (not rec.sellerName or rec.sellerName == "") then
          local s = TA_ReadSellerFromBrowseRow(i)
          if s then rec.sellerName = s; rec.seller = s end
        end
        
        -- Group by item+stack
        local key = string.format("%d:%d", rec.itemId or 0, rec.stackSize or 1)
        capturedByItem[key] = capturedByItem[key] or {}
        table.insert(capturedByItem[key], rec)
      end
    end
    
    -- Cache the captured data
    for key, rows in pairs(capturedByItem) do
      local itemId, stackSize = key:match("(%d+):(%d+)")
      if itemId and stackSize then
        -- Only cache if new or different timestamp
        local lastTime = lastCapture[key] or 0
        if time() - lastTime > 2 then  -- Throttle: same item every 2 seconds
          CacheMarketData(tonumber(itemId), tonumber(stackSize), rows)
          lastCapture[key] = time()
        end
      end
    end
  end)
  
  print("[WhoDAT] Passive market data listener enabled - will capture Auctioneer scans")
end

-- Auction owner list update events
local auctionFrame = CreateFrame("Frame")
auctionFrame:RegisterEvent("AUCTION_OWNED_LIST_UPDATE")
auctionFrame:RegisterEvent("PLAYER_LOGIN")  -- NEW: For addon detection
auctionFrame:SetScript("OnEvent", function(self, event)
    if event == "PLAYER_LOGIN" then
        -- Detect external auction addons
        DetectAuctionAddons()
        
        if activeAuctionSource ~= "WhoDAT_builtin" then
            print(string.format("[WhoDAT] Auction tracking enhanced with: %s", activeAuctionSource))
        end
        -- NEW: Set up passive listener if Auctioneer present
        if activeAuctionSource == "Auctioneer" then
            SetupPassiveListener()
        end
    elseif event == "AUCTION_OWNED_LIST_UPDATE" then
        -- First: Update snapshot for sold/expired detection
        TrackerAuction.UpdateOwnerAuctionsWithChangeDetection()
        
        -- NEW: Also save current auctions to database (with better duplicate handling)
        local myRows, numOwner = TA_ReadOwnerRows()
        
        if numOwner and numOwner > 0 then
            -- Persist to database using improved owner auction upsert
            local realm  = GetRealmName() or "UNKNOWN"
            local faction= UnitFactionGroup("player") or "Neutral"
            local char   = UnitName("player") or "UNKNOWN"
            local key    = realm .. "-" .. faction .. ":" .. char
            
            WhoDAT_AuctionDB = WhoDAT_AuctionDB or {}
            WhoDAT_AuctionDB[key] = WhoDAT_AuctionDB[key] or {}
            local bucket = WhoDAT_AuctionDB[key]
            
            -- Use improved upsert that prevents duplicates
            local newCount = 0
            for i = 1, #myRows do
                local isNew = TA_UpsertOwnerAuction(bucket, myRows[i])
                if isNew then newCount = newCount + 1 end
            end
            
            if newCount > 0 then
                print(string.format("[WhoDAT] Saved %d new auction(s), %d updated", newCount, numOwner - newCount))
            else
                print(string.format("[WhoDAT] Updated %d auction(s)", numOwner))
            end
            
-- NEW: Try to use cached market data (from passive listening)
if CONFIG.AUTO_MARKET_SCAN and activeAuctionSource == "Auctioneer" then
    local marketCount = 0
    local seen = {}
    
    for _, auction in ipairs(myRows) do
        local itemKey = string.format("%d:%d", auction.itemId or 0, auction.stackSize or 1)
        
        if not seen[itemKey] then
            seen[itemKey] = true
            
            -- Try cache first (from Appraiser scan)
            local cachedRows = GetCachedMarketData(auction.itemId, auction.stackSize, 600)  -- 10 min cache
            
            if cachedRows and #cachedRows > 0 then
                local mkt = TrackerAuction.ComputeLowHigh(cachedRows, 3)
                TrackerAuction.PersistMicroSnapshot(auction.itemId, auction.stackSize, mkt, cachedRows)
                marketCount = marketCount + 1
            end
        end
    end
    
    if marketCount > 0 then
        print(string.format("[WhoDAT] Used cached market data for %d item(s)", marketCount))
    end
elseif CONFIG.AUTO_MARKET_SCAN and CONFIG.CONFIRM_MANUAL_SCAN and activeAuctionSource == "WhoDAT_builtin" then
    -- No Auctioneer, offer manual scan
    StaticPopup_Show("WHODAT_MARKET_SCAN_CONFIRM")
end
        end
    end
end)

-- ============================================================================
-- PUBLIC API (NEW: Added for external integration)
-- ============================================================================

function TrackerAuction.GetActiveSource()
  return activeAuctionSource
end

function TrackerAuction.GetAvailableAddons()
  return externalAddonsAvailable
end

function TrackerAuction.ForceBuiltin(enable)
  CONFIG.FORCE_BUILTIN_SCANNING = enable
  DetectAuctionAddons()
end

-- ============================================================================
-- AH SCAN BUTTON (NEW: Add button to Auction House frame)
-- ============================================================================
local scanButton = nil

local function CreateAHScanButton()
  if scanButton then return end
  if not AuctionFrame then return end
  
  -- Create button
  scanButton = CreateFrame("Button", "WhoDAT_AuctionScanButton", AuctionFrame, "UIPanelButtonTemplate")
  scanButton:SetWidth(120)
  scanButton:SetHeight(22)
  scanButton:SetPoint("BOTTOMRIGHT", AuctionFrame, "BOTTOMRIGHT", -10, 10)
  scanButton:SetText("WhoDAT Scan")
  
  scanButton:SetScript("OnClick", function()
    print("[WhoDAT] Starting market scan...")
    TrackerAuction.ScanMyAuctionsWithMarket({}, function(results, stats)
      if stats and stats.error then
        print(string.format("[WhoDAT] Scan failed: %s", stats.error))
      elseif stats and stats.message then
        print(string.format("[WhoDAT] %s", stats.message))
      else
        print(string.format("[WhoDAT] Market scan completed! Scanned %d item(s)", stats and stats.numItems or 0))
      end
    end)
  end)
  
  scanButton:SetScript("OnEnter", function(self)
    GameTooltip:SetOwner(self, "ANCHOR_TOP")
    GameTooltip:SetText("WhoDAT Market Scan", 1, 1, 1)
    GameTooltip:AddLine("Scans your auctions and compares prices with current AH market.", nil, nil, nil, true)
    if activeAuctionSource == "Auctioneer" then
      GameTooltip:AddLine(" ", nil, nil, nil)
      GameTooltip:AddLine("Note: Market data is already being gathered from Auctioneer automatically.", 0.5, 1, 0.5, true)
    end
    GameTooltip:Show()
  end)
  
  scanButton:SetScript("OnLeave", function(self)
    GameTooltip:Hide()
  end)
end

-- Hook to create button when AH opens
local function HookAuctionHouse()
  if not AuctionFrame then
    -- Load Blizzard AH UI if not loaded
    if type(LoadAddOn) == "function" then
      pcall(function() LoadAddOn("Blizzard_AuctionUI") end)
    end
  end
  
  if AuctionFrame and not scanButton then
    -- Wait a bit for frame to be fully initialized
    local f = CreateFrame("Frame")
    local elapsed = 0
    f:SetScript("OnUpdate", function(self, delta)
      elapsed = elapsed + delta
      if elapsed > 0.1 then
        CreateAHScanButton()
        self:SetScript("OnUpdate", nil)
      end
    end)
  end
end

-- Hook when addon loads
if AuctionFrame then
  HookAuctionHouse()
else
  -- Wait for ADDON_LOADED event
  local hookFrame = CreateFrame("Frame")
  hookFrame:RegisterEvent("ADDON_LOADED")
  hookFrame:SetScript("OnEvent", function(self, event, addonName)
    if addonName == "Blizzard_AuctionUI" then
      HookAuctionHouse()
      self:UnregisterEvent("ADDON_LOADED")
    end
  end)
end

-- ============================================================================
-- SLASH COMMANDS (NEW: Added for configuration)
-- ============================================================================
SLASH_WDAUCTION1 = "/wdauction"
SlashCmdList["WDAUCTION"] = function(msg)
  msg = (msg or ""):lower()
  
  if msg == "source" then
    print("=== Auction Data Source ===")
    print(string.format("Active source: %s", activeAuctionSource or "none"))
    print(string.format("Force builtin: %s", tostring(CONFIG.FORCE_BUILTIN_SCANNING)))
    print(string.format("Use Auctioneer for sellers: %s", tostring(CONFIG.USE_AUCTIONEER_FOR_SELLERS)))
    print(string.format("Auto market scan: %s", tostring(CONFIG.AUTO_MARKET_SCAN)))
    print(string.format("Prefer Auctioneer prices: %s", tostring(CONFIG.PREFER_AUCTIONEER_PRICES)))
    print(string.format("Scan confirmation popup: %s", tostring(CONFIG.CONFIRM_MANUAL_SCAN)))
    if #externalAddonsAvailable > 0 then
      print("Available addons:")
      for _, addon in ipairs(externalAddonsAvailable) do
        print(string.format("  - %s", addon))
      end
    else
      print("No external auction addons detected")
    end
    
  elseif msg == "forcebuiltin" then
    CONFIG.FORCE_BUILTIN_SCANNING = not CONFIG.FORCE_BUILTIN_SCANNING
    DetectAuctionAddons()
    print(string.format("[WhoDAT] Force builtin: %s", tostring(CONFIG.FORCE_BUILTIN_SCANNING)))
    print(string.format("[WhoDAT] Now using: %s", activeAuctionSource))
    
  elseif msg == "togglesellers" then
    CONFIG.USE_AUCTIONEER_FOR_SELLERS = not CONFIG.USE_AUCTIONEER_FOR_SELLERS
    print(string.format("[WhoDAT] Auctioneer seller fallback: %s", tostring(CONFIG.USE_AUCTIONEER_FOR_SELLERS)))
    
  elseif msg == "toggleauto" then
    CONFIG.AUTO_MARKET_SCAN = not CONFIG.AUTO_MARKET_SCAN
    print(string.format("[WhoDAT] Auto market scan: %s", tostring(CONFIG.AUTO_MARKET_SCAN)))
    
  elseif msg == "togglepopup" then
    CONFIG.CONFIRM_MANUAL_SCAN = not CONFIG.CONFIRM_MANUAL_SCAN
    print(string.format("[WhoDAT] Scan confirmation popup: %s", tostring(CONFIG.CONFIRM_MANUAL_SCAN)))
    
  elseif msg == "scan" then
    print("[WhoDAT] Starting full auction scan with market data...")
    TrackerAuction.ScanMyAuctionsWithMarket({}, function(results, stats)
      if stats and stats.error then
        print(string.format("[WhoDAT] Scan failed: %s", stats.error))
      elseif stats and stats.message then
        print(string.format("[WhoDAT] %s", stats.message))
      else
        print(string.format("[WhoDAT] Scan completed! Scanned %d item(s)", stats and stats.numItems or 0))
      end
    end)
    
  else
    print("=== WhoDAT Auction Tracker ===")
    print("/wdauction source        - Show data source & settings")
    print("/wdauction scan          - Manual market price scan")
    print("/wdauction forcebuiltin  - Toggle: ignore Auctioneer")
    print("/wdauction togglesellers - Toggle: Auctioneer seller fallback")
    print("/wdauction toggleauto    - Toggle: auto market scan")
    print("/wdauction togglepopup   - Toggle: scan confirmation popup")
    print("")
    print("Auctions auto-save when you view the 'Auctions' tab")
    if activeAuctionSource == "Auctioneer" then
      print("Market data gathered INSTANTLY from Auctioneer (no scanning!)")
    else
      print("Use '/wdauction scan' or the AH button for market data")
    end
  end
end

-- ============================================================================
-- INITIALIZATION
-- ============================================================================
print("[WhoDAT] Auction tracker loaded (with Auctioneer integration support)")
print("[WhoDAT] Auction sold/expired tracking initialized")
print("[WhoDAT] Auctions auto-save when you view the 'Auctions' tab")
print("[WhoDAT] Market prices: " .. (activeAuctionSource == "Auctioneer" and "INSTANT via Auctioneer" or "Manual scan available"))